/*
 *  Arithmetic operators (E5 Sections 11.4.6, 11.4.7, 11.5.1, 11.5.2,
 *  11.5.3, 11.6.1, 11.6.2, 11.6.3).
 *
 *  Operations need to be performed on both constants and variables
 *  (and property accesses etc) because some constant operations are
 *  performed at compile-time.
 *
 *  To distinguish zero signs, the idiom '1 / x' is used.  If x is +0,
 *  this will result in Infinity, and if x is -0, it will result in
 *  -Infinity.
 *
 *  ToNumber conversions are specified in E5 Section 9.3, with the
 *  string grammar given in E5 Section 9.3.1.  The string grammar
 *  is tested in detail in test-conv-tonumber.js.
 */

/* XXX: add IEEE compliance checks (test underlying math library) */
/* XXX: specific integer arithmetic checks for 2^32 and 2^53 range */

/* Shared test values for "matrix tests". */
var testValues = [
    Number.NEGATIVE_INFINITY,
    -1e100,
    -0x100000000,
    -0xffffffff,
    -0x80000000,
    -0x7fffffff,
    -0x12345678,
    -1,
    -1e-100,
    -0,
    +0,
    1e-100,
    +1,
    +0x12345678,
    0x7ffffffff,
    0x800000000,
    0x100000000,
    1e100,
    Number.POSITIVE_INFINITY,
    Number.NaN,

    undefined,
    null,
    "1",
    "+1",
    "-1.5e1",
    true,
    false,
    "1.5e1",
    "2e3",
];

function fmt(v) {
    if (typeof v === 'string') {
        return 'str:' + v;  // happens with '+' operator and two string args
    }
    if (v == 0) {
        if (1 / v < 0) { return '-0'; }
        return '0';
    } else {
        return String(v);
    }
}

/* Expect string is from Rhino. */
/*===
matrix test
0 0 -Infinity -Infinity -Infinity NaN Infinity NaN NaN
0 1 -Infinity -1e+100 -Infinity -Infinity Infinity Infinity NaN
0 2 -Infinity -4294967296 -Infinity -Infinity Infinity Infinity NaN
0 3 -Infinity -4294967295 -Infinity -Infinity Infinity Infinity NaN
0 4 -Infinity -2147483648 -Infinity -Infinity Infinity Infinity NaN
0 5 -Infinity -2147483647 -Infinity -Infinity Infinity Infinity NaN
0 6 -Infinity -305419896 -Infinity -Infinity Infinity Infinity NaN
0 7 -Infinity -1 -Infinity -Infinity Infinity Infinity NaN
0 8 -Infinity -1e-100 -Infinity -Infinity Infinity Infinity NaN
0 9 -Infinity -0 -Infinity -Infinity NaN Infinity NaN
0 10 -Infinity 0 -Infinity -Infinity NaN -Infinity NaN
0 11 -Infinity 1e-100 -Infinity -Infinity -Infinity -Infinity NaN
0 12 -Infinity 1 -Infinity -Infinity -Infinity -Infinity NaN
0 13 -Infinity 305419896 -Infinity -Infinity -Infinity -Infinity NaN
0 14 -Infinity 34359738367 -Infinity -Infinity -Infinity -Infinity NaN
0 15 -Infinity 34359738368 -Infinity -Infinity -Infinity -Infinity NaN
0 16 -Infinity 4294967296 -Infinity -Infinity -Infinity -Infinity NaN
0 17 -Infinity 1e+100 -Infinity -Infinity -Infinity -Infinity NaN
0 18 -Infinity Infinity NaN -Infinity -Infinity NaN NaN
0 19 -Infinity NaN NaN NaN NaN NaN NaN
0 20 -Infinity undefined NaN NaN NaN NaN NaN
0 21 -Infinity null -Infinity -Infinity NaN -Infinity NaN
0 22 -Infinity str:1 str:-Infinity1 -Infinity -Infinity -Infinity NaN
0 23 -Infinity str:+1 str:-Infinity+1 -Infinity -Infinity -Infinity NaN
0 24 -Infinity str:-1.5e1 str:-Infinity-1.5e1 -Infinity Infinity Infinity NaN
0 25 -Infinity true -Infinity -Infinity -Infinity -Infinity NaN
0 26 -Infinity 0 -Infinity -Infinity NaN -Infinity NaN
0 27 -Infinity str:1.5e1 str:-Infinity1.5e1 -Infinity -Infinity -Infinity NaN
0 28 -Infinity str:2e3 str:-Infinity2e3 -Infinity -Infinity -Infinity NaN
1 0 -1e+100 -Infinity -Infinity Infinity Infinity 0 -1e+100
1 1 -1e+100 -1e+100 -2e+100 0 1e+200 1 -0
1 2 -1e+100 -4294967296 -1e+100 -1e+100 4.294967296e+109 2.3283064365386963e+90 -0
1 3 -1e+100 -4294967295 -1e+100 -1e+100 4.294967295e+109 2.3283064370807973e+90 -707110669
1 4 -1e+100 -2147483648 -1e+100 -1e+100 2.147483648e+109 4.656612873077393e+90 -0
1 5 -1e+100 -2147483647 -1e+100 -1e+100 2.147483647e+109 4.656612875245797e+90 -1265806766
1 6 -1e+100 -305419896 -1e+100 -1e+100 3.05419896e+108 3.274180932862344e+91 -43164928
1 7 -1e+100 -1 -1e+100 -1e+100 1e+100 1e+100 -0
1 8 -1e+100 -1e-100 -1e+100 -1e+100 1 1e+200 -9.885752855515831e-101
1 9 -1e+100 -0 -1e+100 -1e+100 0 Infinity NaN
1 10 -1e+100 0 -1e+100 -1e+100 -0 -Infinity NaN
1 11 -1e+100 1e-100 -1e+100 -1e+100 -1 -1e+200 -9.885752855515831e-101
1 12 -1e+100 1 -1e+100 -1e+100 -1e+100 -1e+100 -0
1 13 -1e+100 305419896 -1e+100 -1e+100 -3.05419896e+108 -3.274180932862344e+91 -43164928
1 14 -1e+100 34359738367 -1e+100 -1e+100 -3.4359738367e+110 -2.910383045758074e+89 -22105492658
1 15 -1e+100 34359738368 -1e+100 -1e+100 -3.4359738368e+110 -2.9103830456733704e+89 -0
1 16 -1e+100 4294967296 -1e+100 -1e+100 -4.294967296e+109 -2.3283064365386963e+90 -0
1 17 -1e+100 1e+100 0 -2e+100 -1e+200 -1 -0
1 18 -1e+100 Infinity Infinity -Infinity -Infinity -0 -1e+100
1 19 -1e+100 NaN NaN NaN NaN NaN NaN
1 20 -1e+100 undefined NaN NaN NaN NaN NaN
1 21 -1e+100 null -1e+100 -1e+100 -0 -Infinity NaN
1 22 -1e+100 str:1 str:-1e+1001 -1e+100 -1e+100 -1e+100 -0
1 23 -1e+100 str:+1 str:-1e+100+1 -1e+100 -1e+100 -1e+100 -0
1 24 -1e+100 str:-1.5e1 str:-1e+100-1.5e1 -1e+100 1.5e+101 6.666666666666666e+98 -4
1 25 -1e+100 true -1e+100 -1e+100 -1e+100 -1e+100 -0
1 26 -1e+100 0 -1e+100 -1e+100 -0 -Infinity NaN
1 27 -1e+100 str:1.5e1 str:-1e+1001.5e1 -1e+100 -1.5e+101 -6.666666666666666e+98 -4
1 28 -1e+100 str:2e3 str:-1e+1002e3 -1e+100 -2e+103 -5e+96 -1104
2 0 -4294967296 -Infinity -Infinity Infinity Infinity 0 -4294967296
2 1 -4294967296 -1e+100 -1e+100 1e+100 4.294967296e+109 4.294967296e-91 -4294967296
2 2 -4294967296 -4294967296 -8589934592 0 18446744073709552000 1 -0
2 3 -4294967296 -4294967295 -8589934591 -1 18446744069414584000 1.0000000002328306 -1
2 4 -4294967296 -2147483648 -6442450944 -2147483648 9223372036854776000 2 -0
2 5 -4294967296 -2147483647 -6442450943 -2147483649 9223372032559809000 2.0000000009313226 -2
2 6 -4294967296 -305419896 -4600387192 -3989547400 1311768464867721200 14.062500027830538 -19088752
2 7 -4294967296 -1 -4294967297 -4294967295 4294967296 4294967296 -0
2 8 -4294967296 -1e-100 -4294967296 -4294967296 4.294967296e-91 4.294967296e+109 -5.245765197922255e-101
2 9 -4294967296 -0 -4294967296 -4294967296 0 Infinity NaN
2 10 -4294967296 0 -4294967296 -4294967296 -0 -Infinity NaN
2 11 -4294967296 1e-100 -4294967296 -4294967296 -4.294967296e-91 -4.294967296e+109 -5.245765197922255e-101
2 12 -4294967296 1 -4294967295 -4294967297 -4294967296 -4294967296 -0
2 13 -4294967296 305419896 -3989547400 -4600387192 -1311768464867721200 -14.062500027830538 -19088752
2 14 -4294967296 34359738367 30064771071 -38654705663 -147573952585381450000 -0.12500000000363798 -4294967296
2 15 -4294967296 34359738368 30064771072 -38654705664 -147573952589676410000 -0.125 -4294967296
2 16 -4294967296 4294967296 0 -8589934592 -18446744073709552000 -1 -0
2 17 -4294967296 1e+100 1e+100 -1e+100 -4.294967296e+109 -4.294967296e-91 -4294967296
2 18 -4294967296 Infinity Infinity -Infinity -Infinity -0 -4294967296
2 19 -4294967296 NaN NaN NaN NaN NaN NaN
2 20 -4294967296 undefined NaN NaN NaN NaN NaN
2 21 -4294967296 null -4294967296 -4294967296 -0 -Infinity NaN
2 22 -4294967296 str:1 str:-42949672961 -4294967297 -4294967296 -4294967296 -0
2 23 -4294967296 str:+1 str:-4294967296+1 -4294967297 -4294967296 -4294967296 -0
2 24 -4294967296 str:-1.5e1 str:-4294967296-1.5e1 -4294967281 64424509440 286331153.06666666 -1
2 25 -4294967296 true -4294967295 -4294967297 -4294967296 -4294967296 -0
2 26 -4294967296 0 -4294967296 -4294967296 -0 -Infinity NaN
2 27 -4294967296 str:1.5e1 str:-42949672961.5e1 -4294967311 -64424509440 -286331153.06666666 -1
2 28 -4294967296 str:2e3 str:-42949672962e3 -4294969296 -8589934592000 -2147483.648 -1296
3 0 -4294967295 -Infinity -Infinity Infinity Infinity 0 -4294967295
3 1 -4294967295 -1e+100 -1e+100 1e+100 4.294967295e+109 4.294967295e-91 -4294967295
3 2 -4294967295 -4294967296 -8589934591 1 18446744069414584000 0.9999999997671694 -4294967295
3 3 -4294967295 -4294967295 -8589934590 0 18446744065119617000 1 -0
3 4 -4294967295 -2147483648 -6442450943 -2147483647 9223372034707292000 1.9999999995343387 -2147483647
3 5 -4294967295 -2147483647 -6442450942 -2147483648 9223372030412325000 2.0000000004656613 -1
3 6 -4294967295 -305419896 -4600387191 -3989547399 1311768464562301400 14.062500024556357 -19088751
3 7 -4294967295 -1 -4294967296 -4294967294 4294967295 4294967295 -0
3 8 -4294967295 -1e-100 -4294967295 -4294967295 4.294967295e-91 4.294967295e+109 -7.166912733187267e-101
3 9 -4294967295 -0 -4294967295 -4294967295 0 Infinity NaN
3 10 -4294967295 0 -4294967295 -4294967295 -0 -Infinity NaN
3 11 -4294967295 1e-100 -4294967295 -4294967295 -4.294967295e-91 -4.294967295e+109 -7.166912733187267e-101
3 12 -4294967295 1 -4294967294 -4294967296 -4294967295 -4294967295 -0
3 13 -4294967295 305419896 -3989547399 -4600387191 -1311768464562301400 -14.062500024556357 -19088751
3 14 -4294967295 34359738367 30064771072 -38654705662 -147573952551021700000 -0.12499999997453415 -4294967295
3 15 -4294967295 34359738368 30064771073 -38654705663 -147573952555316670000 -0.12499999997089617 -4294967295
3 16 -4294967295 4294967296 1 -8589934591 -18446744069414584000 -0.9999999997671694 -4294967295
3 17 -4294967295 1e+100 1e+100 -1e+100 -4.294967295e+109 -4.294967295e-91 -4294967295
3 18 -4294967295 Infinity Infinity -Infinity -Infinity -0 -4294967295
3 19 -4294967295 NaN NaN NaN NaN NaN NaN
3 20 -4294967295 undefined NaN NaN NaN NaN NaN
3 21 -4294967295 null -4294967295 -4294967295 -0 -Infinity NaN
3 22 -4294967295 str:1 str:-42949672951 -4294967296 -4294967295 -4294967295 -0
3 23 -4294967295 str:+1 str:-4294967295+1 -4294967296 -4294967295 -4294967295 -0
3 24 -4294967295 str:-1.5e1 str:-4294967295-1.5e1 -4294967280 64424509425 286331153 -0
3 25 -4294967295 true -4294967294 -4294967296 -4294967295 -4294967295 -0
3 26 -4294967295 0 -4294967295 -4294967295 -0 -Infinity NaN
3 27 -4294967295 str:1.5e1 str:-42949672951.5e1 -4294967310 -64424509425 -286331153 -0
3 28 -4294967295 str:2e3 str:-42949672952e3 -4294969295 -8589934590000 -2147483.6475 -1295
4 0 -2147483648 -Infinity -Infinity Infinity Infinity 0 -2147483648
4 1 -2147483648 -1e+100 -1e+100 1e+100 2.147483648e+109 2.147483648e-91 -2147483648
4 2 -2147483648 -4294967296 -6442450944 2147483648 9223372036854776000 0.5 -2147483648
4 3 -2147483648 -4294967295 -6442450943 2147483647 9223372034707292000 0.5000000001164153 -2147483648
4 4 -2147483648 -2147483648 -4294967296 0 4611686018427388000 1 -0
4 5 -2147483648 -2147483647 -4294967295 -1 4611686016279904000 1.0000000004656613 -1
4 6 -2147483648 -305419896 -2452903544 -1842063752 655884232433860600 7.031250013915269 -9544376
4 7 -2147483648 -1 -2147483649 -2147483647 2147483648 2147483648 -0
4 8 -2147483648 -1e-100 -2147483648 -2147483648 2.147483648e-91 2.147483648e+109 -2.6228825989611275e-101
4 9 -2147483648 -0 -2147483648 -2147483648 0 Infinity NaN
4 10 -2147483648 0 -2147483648 -2147483648 -0 -Infinity NaN
4 11 -2147483648 1e-100 -2147483648 -2147483648 -2.147483648e-91 -2.147483648e+109 -2.6228825989611275e-101
4 12 -2147483648 1 -2147483647 -2147483649 -2147483648 -2147483648 -0
4 13 -2147483648 305419896 -1842063752 -2452903544 -655884232433860600 -7.031250013915269 -9544376
4 14 -2147483648 34359738367 32212254719 -36507222015 -73786976292690720000 -0.06250000000181899 -2147483648
4 15 -2147483648 34359738368 32212254720 -36507222016 -73786976294838210000 -0.0625 -2147483648
4 16 -2147483648 4294967296 2147483648 -6442450944 -9223372036854776000 -0.5 -2147483648
4 17 -2147483648 1e+100 1e+100 -1e+100 -2.147483648e+109 -2.147483648e-91 -2147483648
4 18 -2147483648 Infinity Infinity -Infinity -Infinity -0 -2147483648
4 19 -2147483648 NaN NaN NaN NaN NaN NaN
4 20 -2147483648 undefined NaN NaN NaN NaN NaN
4 21 -2147483648 null -2147483648 -2147483648 -0 -Infinity NaN
4 22 -2147483648 str:1 str:-21474836481 -2147483649 -2147483648 -2147483648 -0
4 23 -2147483648 str:+1 str:-2147483648+1 -2147483649 -2147483648 -2147483648 -0
4 24 -2147483648 str:-1.5e1 str:-2147483648-1.5e1 -2147483633 32212254720 143165576.53333333 -8
4 25 -2147483648 true -2147483647 -2147483649 -2147483648 -2147483648 -0
4 26 -2147483648 0 -2147483648 -2147483648 -0 -Infinity NaN
4 27 -2147483648 str:1.5e1 str:-21474836481.5e1 -2147483663 -32212254720 -143165576.53333333 -8
4 28 -2147483648 str:2e3 str:-21474836482e3 -2147485648 -4294967296000 -1073741.824 -1648
5 0 -2147483647 -Infinity -Infinity Infinity Infinity 0 -2147483647
5 1 -2147483647 -1e+100 -1e+100 1e+100 2.147483647e+109 2.147483647e-91 -2147483647
5 2 -2147483647 -4294967296 -6442450943 2147483649 9223372032559809000 0.49999999976716936 -2147483647
5 3 -2147483647 -4294967295 -6442450942 2147483648 9223372030412325000 0.4999999998835847 -2147483647
5 4 -2147483647 -2147483648 -4294967295 1 4611686016279904000 0.9999999995343387 -2147483647
5 5 -2147483647 -2147483647 -4294967294 0 4611686014132420600 1 -0
5 6 -2147483647 -305419896 -2452903543 -1842063751 655884232128440700 7.031250010641088 -9544375
5 7 -2147483647 -1 -2147483648 -2147483646 2147483647 2147483647 -0
5 8 -2147483647 -1e-100 -2147483647 -2147483647 2.147483647e-91 2.147483647e+109 -4.54403013422614e-101
5 9 -2147483647 -0 -2147483647 -2147483647 0 Infinity NaN
5 10 -2147483647 0 -2147483647 -2147483647 -0 -Infinity NaN
5 11 -2147483647 1e-100 -2147483647 -2147483647 -2.147483647e-91 -2.147483647e+109 -4.54403013422614e-101
5 12 -2147483647 1 -2147483646 -2147483648 -2147483647 -2147483647 -0
5 13 -2147483647 305419896 -1842063751 -2452903543 -655884232128440700 -7.031250010641088 -9544375
5 14 -2147483647 34359738367 32212254720 -36507222014 -73786976258330984000 -0.06249999997271516 -2147483647
5 15 -2147483647 34359738368 32212254721 -36507222015 -73786976260478470000 -0.06249999997089617 -2147483647
5 16 -2147483647 4294967296 2147483649 -6442450943 -9223372032559809000 -0.49999999976716936 -2147483647
5 17 -2147483647 1e+100 1e+100 -1e+100 -2.147483647e+109 -2.147483647e-91 -2147483647
5 18 -2147483647 Infinity Infinity -Infinity -Infinity -0 -2147483647
5 19 -2147483647 NaN NaN NaN NaN NaN NaN
5 20 -2147483647 undefined NaN NaN NaN NaN NaN
5 21 -2147483647 null -2147483647 -2147483647 -0 -Infinity NaN
5 22 -2147483647 str:1 str:-21474836471 -2147483648 -2147483647 -2147483647 -0
5 23 -2147483647 str:+1 str:-2147483647+1 -2147483648 -2147483647 -2147483647 -0
5 24 -2147483647 str:-1.5e1 str:-2147483647-1.5e1 -2147483632 32212254705 143165576.46666667 -7
5 25 -2147483647 true -2147483646 -2147483648 -2147483647 -2147483647 -0
5 26 -2147483647 0 -2147483647 -2147483647 -0 -Infinity NaN
5 27 -2147483647 str:1.5e1 str:-21474836471.5e1 -2147483662 -32212254705 -143165576.46666667 -7
5 28 -2147483647 str:2e3 str:-21474836472e3 -2147485647 -4294967294000 -1073741.8235 -1647
6 0 -305419896 -Infinity -Infinity Infinity Infinity 0 -305419896
6 1 -305419896 -1e+100 -1e+100 1e+100 3.05419896e+108 3.05419896e-92 -305419896
6 2 -305419896 -4294967296 -4600387192 3989547400 1311768464867721200 0.07111111097037792 -305419896
6 3 -305419896 -4294967295 -4600387191 3989547399 1311768464562301400 0.07111111098693476 -305419896
6 4 -305419896 -2147483648 -2452903544 1842063752 655884232433860600 0.14222222194075584 -305419896
6 5 -305419896 -2147483647 -2452903543 1842063751 655884232128440700 0.14222222200698323 -305419896
6 6 -305419896 -305419896 -610839792 0 93281312872650820 1 -0
6 7 -305419896 -1 -305419897 -305419895 305419896 305419896 -0
6 8 -305419896 -1e-100 -305419896 -305419896 3.05419896e-92 3.05419896e+108 -9.578703590945768e-101
6 9 -305419896 -0 -305419896 -305419896 0 Infinity NaN
6 10 -305419896 0 -305419896 -305419896 -0 -Infinity NaN
6 11 -305419896 1e-100 -305419896 -305419896 -3.05419896e-92 -3.05419896e+108 -9.578703590945768e-101
6 12 -305419896 1 -305419895 -305419897 -305419896 -305419896 -0
6 13 -305419896 305419896 0 -610839792 -93281312872650820 -1 -0
6 14 -305419896 34359738367 34054318471 -34665158263 -10494147718636350000 -0.008888888871555941 -305419896
6 15 -305419896 34359738368 34054318472 -34665158264 -10494147718941770000 -0.00888888887129724 -305419896
6 16 -305419896 4294967296 3989547400 -4600387192 -1311768464867721200 -0.07111111097037792 -305419896
6 17 -305419896 1e+100 1e+100 -1e+100 -3.05419896e+108 -3.05419896e-92 -305419896
6 18 -305419896 Infinity Infinity -Infinity -Infinity -0 -305419896
6 19 -305419896 NaN NaN NaN NaN NaN NaN
6 20 -305419896 undefined NaN NaN NaN NaN NaN
6 21 -305419896 null -305419896 -305419896 -0 -Infinity NaN
6 22 -305419896 str:1 str:-3054198961 -305419897 -305419896 -305419896 -0
6 23 -305419896 str:+1 str:-305419896+1 -305419897 -305419896 -305419896 -0
6 24 -305419896 str:-1.5e1 str:-305419896-1.5e1 -305419881 4581298440 20361326.4 -6
6 25 -305419896 true -305419895 -305419897 -305419896 -305419896 -0
6 26 -305419896 0 -305419896 -305419896 -0 -Infinity NaN
6 27 -305419896 str:1.5e1 str:-3054198961.5e1 -305419911 -4581298440 -20361326.4 -6
6 28 -305419896 str:2e3 str:-3054198962e3 -305421896 -610839792000 -152709.948 -1896
7 0 -1 -Infinity -Infinity Infinity Infinity 0 -1
7 1 -1 -1e+100 -1e+100 1e+100 1e+100 1e-100 -1
7 2 -1 -4294967296 -4294967297 4294967295 4294967296 2.3283064365386963e-10 -1
7 3 -1 -4294967295 -4294967296 4294967294 4294967295 2.3283064370807974e-10 -1
7 4 -1 -2147483648 -2147483649 2147483647 2147483648 4.656612873077393e-10 -1
7 5 -1 -2147483647 -2147483648 2147483646 2147483647 4.656612875245797e-10 -1
7 6 -1 -305419896 -305419897 305419895 305419896 3.2741809328623437e-9 -1
7 7 -1 -1 -2 0 1 1 -0
7 8 -1 -1e-100 -1 -1 1e-100 1e+100 -8.078852464734988e-101
7 9 -1 -0 -1 -1 0 Infinity NaN
7 10 -1 0 -1 -1 -0 -Infinity NaN
7 11 -1 1e-100 -1 -1 -1e-100 -1e+100 -8.078852464734988e-101
7 12 -1 1 0 -2 -1 -1 -0
7 13 -1 305419896 305419895 -305419897 -305419896 -3.2741809328623437e-9 -1
7 14 -1 34359738367 34359738366 -34359738368 -34359738367 -2.9103830457580737e-11 -1
7 15 -1 34359738368 34359738367 -34359738369 -34359738368 -2.9103830456733704e-11 -1
7 16 -1 4294967296 4294967295 -4294967297 -4294967296 -2.3283064365386963e-10 -1
7 17 -1 1e+100 1e+100 -1e+100 -1e+100 -1e-100 -1
7 18 -1 Infinity Infinity -Infinity -Infinity -0 -1
7 19 -1 NaN NaN NaN NaN NaN NaN
7 20 -1 undefined NaN NaN NaN NaN NaN
7 21 -1 null -1 -1 -0 -Infinity NaN
7 22 -1 str:1 str:-11 -2 -1 -1 -0
7 23 -1 str:+1 str:-1+1 -2 -1 -1 -0
7 24 -1 str:-1.5e1 str:-1-1.5e1 14 15 0.06666666666666667 -1
7 25 -1 true 0 -2 -1 -1 -0
7 26 -1 0 -1 -1 -0 -Infinity NaN
7 27 -1 str:1.5e1 str:-11.5e1 -16 -15 -0.06666666666666667 -1
7 28 -1 str:2e3 str:-12e3 -2001 -2000 -0.0005 -1
8 0 -1e-100 -Infinity -Infinity Infinity Infinity 0 -1e-100
8 1 -1e-100 -1e+100 -1e+100 1e+100 1 1e-200 -1e-100
8 2 -1e-100 -4294967296 -4294967296 4294967296 4.294967296e-91 2.3283064365386963e-110 -1e-100
8 3 -1e-100 -4294967295 -4294967295 4294967295 4.294967295e-91 2.3283064370807975e-110 -1e-100
8 4 -1e-100 -2147483648 -2147483648 2147483648 2.147483648e-91 4.6566128730773927e-110 -1e-100
8 5 -1e-100 -2147483647 -2147483647 2147483647 2.147483647e-91 4.656612875245797e-110 -1e-100
8 6 -1e-100 -305419896 -305419896 305419896 3.05419896e-92 3.274180932862344e-109 -1e-100
8 7 -1e-100 -1 -1 1 1e-100 1e-100 -1e-100
8 8 -1e-100 -1e-100 -2e-100 0 1e-200 1 -0
8 9 -1e-100 -0 -1e-100 -1e-100 0 Infinity NaN
8 10 -1e-100 0 -1e-100 -1e-100 -0 -Infinity NaN
8 11 -1e-100 1e-100 0 -2e-100 -1e-200 -1 -0
8 12 -1e-100 1 1 -1 -1e-100 -1e-100 -1e-100
8 13 -1e-100 305419896 305419896 -305419896 -3.05419896e-92 -3.274180932862344e-109 -1e-100
8 14 -1e-100 34359738367 34359738367 -34359738367 -3.4359738367e-90 -2.9103830457580738e-111 -1e-100
8 15 -1e-100 34359738368 34359738368 -34359738368 -3.4359738368e-90 -2.9103830456733704e-111 -1e-100
8 16 -1e-100 4294967296 4294967296 -4294967296 -4.294967296e-91 -2.3283064365386963e-110 -1e-100
8 17 -1e-100 1e+100 1e+100 -1e+100 -1 -1e-200 -1e-100
8 18 -1e-100 Infinity Infinity -Infinity -Infinity -0 -1e-100
8 19 -1e-100 NaN NaN NaN NaN NaN NaN
8 20 -1e-100 undefined NaN NaN NaN NaN NaN
8 21 -1e-100 null -1e-100 -1e-100 -0 -Infinity NaN
8 22 -1e-100 str:1 str:-1e-1001 -1 -1e-100 -1e-100 -1e-100
8 23 -1e-100 str:+1 str:-1e-100+1 -1 -1e-100 -1e-100 -1e-100
8 24 -1e-100 str:-1.5e1 str:-1e-100-1.5e1 15 1.5e-99 6.666666666666667e-102 -1e-100
8 25 -1e-100 true 1 -1 -1e-100 -1e-100 -1e-100
8 26 -1e-100 0 -1e-100 -1e-100 -0 -Infinity NaN
8 27 -1e-100 str:1.5e1 str:-1e-1001.5e1 -15 -1.5e-99 -6.666666666666667e-102 -1e-100
8 28 -1e-100 str:2e3 str:-1e-1002e3 -2000 -2e-97 -5.0000000000000004e-104 -1e-100
9 0 -0 -Infinity -Infinity Infinity NaN 0 -0
9 1 -0 -1e+100 -1e+100 1e+100 0 0 -0
9 2 -0 -4294967296 -4294967296 4294967296 0 0 -0
9 3 -0 -4294967295 -4294967295 4294967295 0 0 -0
9 4 -0 -2147483648 -2147483648 2147483648 0 0 -0
9 5 -0 -2147483647 -2147483647 2147483647 0 0 -0
9 6 -0 -305419896 -305419896 305419896 0 0 -0
9 7 -0 -1 -1 1 0 0 -0
9 8 -0 -1e-100 -1e-100 1e-100 0 0 -0
9 9 -0 -0 -0 0 0 NaN NaN
9 10 -0 0 0 -0 -0 NaN NaN
9 11 -0 1e-100 1e-100 -1e-100 -0 -0 -0
9 12 -0 1 1 -1 -0 -0 -0
9 13 -0 305419896 305419896 -305419896 -0 -0 -0
9 14 -0 34359738367 34359738367 -34359738367 -0 -0 -0
9 15 -0 34359738368 34359738368 -34359738368 -0 -0 -0
9 16 -0 4294967296 4294967296 -4294967296 -0 -0 -0
9 17 -0 1e+100 1e+100 -1e+100 -0 -0 -0
9 18 -0 Infinity Infinity -Infinity NaN -0 -0
9 19 -0 NaN NaN NaN NaN NaN NaN
9 20 -0 undefined NaN NaN NaN NaN NaN
9 21 -0 null 0 -0 -0 NaN NaN
9 22 -0 str:1 str:01 -1 -0 -0 -0
9 23 -0 str:+1 str:0+1 -1 -0 -0 -0
9 24 -0 str:-1.5e1 str:0-1.5e1 15 0 0 -0
9 25 -0 true 1 -1 -0 -0 -0
9 26 -0 0 0 -0 -0 NaN NaN
9 27 -0 str:1.5e1 str:01.5e1 -15 -0 -0 -0
9 28 -0 str:2e3 str:02e3 -2000 -0 -0 -0
10 0 0 -Infinity -Infinity Infinity NaN -0 0
10 1 0 -1e+100 -1e+100 1e+100 -0 -0 0
10 2 0 -4294967296 -4294967296 4294967296 -0 -0 0
10 3 0 -4294967295 -4294967295 4294967295 -0 -0 0
10 4 0 -2147483648 -2147483648 2147483648 -0 -0 0
10 5 0 -2147483647 -2147483647 2147483647 -0 -0 0
10 6 0 -305419896 -305419896 305419896 -0 -0 0
10 7 0 -1 -1 1 -0 -0 0
10 8 0 -1e-100 -1e-100 1e-100 -0 -0 0
10 9 0 -0 0 0 -0 NaN NaN
10 10 0 0 0 0 0 NaN NaN
10 11 0 1e-100 1e-100 -1e-100 0 0 0
10 12 0 1 1 -1 0 0 0
10 13 0 305419896 305419896 -305419896 0 0 0
10 14 0 34359738367 34359738367 -34359738367 0 0 0
10 15 0 34359738368 34359738368 -34359738368 0 0 0
10 16 0 4294967296 4294967296 -4294967296 0 0 0
10 17 0 1e+100 1e+100 -1e+100 0 0 0
10 18 0 Infinity Infinity -Infinity NaN 0 0
10 19 0 NaN NaN NaN NaN NaN NaN
10 20 0 undefined NaN NaN NaN NaN NaN
10 21 0 null 0 0 0 NaN NaN
10 22 0 str:1 str:01 -1 0 0 0
10 23 0 str:+1 str:0+1 -1 0 0 0
10 24 0 str:-1.5e1 str:0-1.5e1 15 -0 -0 0
10 25 0 true 1 -1 0 0 0
10 26 0 0 0 0 0 NaN NaN
10 27 0 str:1.5e1 str:01.5e1 -15 0 0 0
10 28 0 str:2e3 str:02e3 -2000 0 0 0
11 0 1e-100 -Infinity -Infinity Infinity -Infinity -0 1e-100
11 1 1e-100 -1e+100 -1e+100 1e+100 -1 -1e-200 1e-100
11 2 1e-100 -4294967296 -4294967296 4294967296 -4.294967296e-91 -2.3283064365386963e-110 1e-100
11 3 1e-100 -4294967295 -4294967295 4294967295 -4.294967295e-91 -2.3283064370807975e-110 1e-100
11 4 1e-100 -2147483648 -2147483648 2147483648 -2.147483648e-91 -4.6566128730773927e-110 1e-100
11 5 1e-100 -2147483647 -2147483647 2147483647 -2.147483647e-91 -4.656612875245797e-110 1e-100
11 6 1e-100 -305419896 -305419896 305419896 -3.05419896e-92 -3.274180932862344e-109 1e-100
11 7 1e-100 -1 -1 1 -1e-100 -1e-100 1e-100
11 8 1e-100 -1e-100 0 2e-100 -1e-200 -1 0
11 9 1e-100 -0 1e-100 1e-100 -0 -Infinity NaN
11 10 1e-100 0 1e-100 1e-100 0 Infinity NaN
11 11 1e-100 1e-100 2e-100 0 1e-200 1 0
11 12 1e-100 1 1 -1 1e-100 1e-100 1e-100
11 13 1e-100 305419896 305419896 -305419896 3.05419896e-92 3.274180932862344e-109 1e-100
11 14 1e-100 34359738367 34359738367 -34359738367 3.4359738367e-90 2.9103830457580738e-111 1e-100
11 15 1e-100 34359738368 34359738368 -34359738368 3.4359738368e-90 2.9103830456733704e-111 1e-100
11 16 1e-100 4294967296 4294967296 -4294967296 4.294967296e-91 2.3283064365386963e-110 1e-100
11 17 1e-100 1e+100 1e+100 -1e+100 1 1e-200 1e-100
11 18 1e-100 Infinity Infinity -Infinity Infinity 0 1e-100
11 19 1e-100 NaN NaN NaN NaN NaN NaN
11 20 1e-100 undefined NaN NaN NaN NaN NaN
11 21 1e-100 null 1e-100 1e-100 0 Infinity NaN
11 22 1e-100 str:1 str:1e-1001 -1 1e-100 1e-100 1e-100
11 23 1e-100 str:+1 str:1e-100+1 -1 1e-100 1e-100 1e-100
11 24 1e-100 str:-1.5e1 str:1e-100-1.5e1 15 -1.5e-99 -6.666666666666667e-102 1e-100
11 25 1e-100 true 1 -1 1e-100 1e-100 1e-100
11 26 1e-100 0 1e-100 1e-100 0 Infinity NaN
11 27 1e-100 str:1.5e1 str:1e-1001.5e1 -15 1.5e-99 6.666666666666667e-102 1e-100
11 28 1e-100 str:2e3 str:1e-1002e3 -2000 2e-97 5.0000000000000004e-104 1e-100
12 0 1 -Infinity -Infinity Infinity -Infinity -0 1
12 1 1 -1e+100 -1e+100 1e+100 -1e+100 -1e-100 1
12 2 1 -4294967296 -4294967295 4294967297 -4294967296 -2.3283064365386963e-10 1
12 3 1 -4294967295 -4294967294 4294967296 -4294967295 -2.3283064370807974e-10 1
12 4 1 -2147483648 -2147483647 2147483649 -2147483648 -4.656612873077393e-10 1
12 5 1 -2147483647 -2147483646 2147483648 -2147483647 -4.656612875245797e-10 1
12 6 1 -305419896 -305419895 305419897 -305419896 -3.2741809328623437e-9 1
12 7 1 -1 0 2 -1 -1 0
12 8 1 -1e-100 1 1 -1e-100 -1e+100 8.078852464734988e-101
12 9 1 -0 1 1 -0 -Infinity NaN
12 10 1 0 1 1 0 Infinity NaN
12 11 1 1e-100 1 1 1e-100 1e+100 8.078852464734988e-101
12 12 1 1 2 0 1 1 0
12 13 1 305419896 305419897 -305419895 305419896 3.2741809328623437e-9 1
12 14 1 34359738367 34359738368 -34359738366 34359738367 2.9103830457580737e-11 1
12 15 1 34359738368 34359738369 -34359738367 34359738368 2.9103830456733704e-11 1
12 16 1 4294967296 4294967297 -4294967295 4294967296 2.3283064365386963e-10 1
12 17 1 1e+100 1e+100 -1e+100 1e+100 1e-100 1
12 18 1 Infinity Infinity -Infinity Infinity 0 1
12 19 1 NaN NaN NaN NaN NaN NaN
12 20 1 undefined NaN NaN NaN NaN NaN
12 21 1 null 1 1 0 Infinity NaN
12 22 1 str:1 str:11 0 1 1 0
12 23 1 str:+1 str:1+1 0 1 1 0
12 24 1 str:-1.5e1 str:1-1.5e1 16 -15 -0.06666666666666667 1
12 25 1 true 2 0 1 1 0
12 26 1 0 1 1 0 Infinity NaN
12 27 1 str:1.5e1 str:11.5e1 -14 15 0.06666666666666667 1
12 28 1 str:2e3 str:12e3 -1999 2000 0.0005 1
13 0 305419896 -Infinity -Infinity Infinity -Infinity -0 305419896
13 1 305419896 -1e+100 -1e+100 1e+100 -3.05419896e+108 -3.05419896e-92 305419896
13 2 305419896 -4294967296 -3989547400 4600387192 -1311768464867721200 -0.07111111097037792 305419896
13 3 305419896 -4294967295 -3989547399 4600387191 -1311768464562301400 -0.07111111098693476 305419896
13 4 305419896 -2147483648 -1842063752 2452903544 -655884232433860600 -0.14222222194075584 305419896
13 5 305419896 -2147483647 -1842063751 2452903543 -655884232128440700 -0.14222222200698323 305419896
13 6 305419896 -305419896 0 610839792 -93281312872650820 -1 0
13 7 305419896 -1 305419895 305419897 -305419896 -305419896 0
13 8 305419896 -1e-100 305419896 305419896 -3.05419896e-92 -3.05419896e+108 9.578703590945768e-101
13 9 305419896 -0 305419896 305419896 -0 -Infinity NaN
13 10 305419896 0 305419896 305419896 0 Infinity NaN
13 11 305419896 1e-100 305419896 305419896 3.05419896e-92 3.05419896e+108 9.578703590945768e-101
13 12 305419896 1 305419897 305419895 305419896 305419896 0
13 13 305419896 305419896 610839792 0 93281312872650820 1 0
13 14 305419896 34359738367 34665158263 -34054318471 10494147718636350000 0.008888888871555941 305419896
13 15 305419896 34359738368 34665158264 -34054318472 10494147718941770000 0.00888888887129724 305419896
13 16 305419896 4294967296 4600387192 -3989547400 1311768464867721200 0.07111111097037792 305419896
13 17 305419896 1e+100 1e+100 -1e+100 3.05419896e+108 3.05419896e-92 305419896
13 18 305419896 Infinity Infinity -Infinity Infinity 0 305419896
13 19 305419896 NaN NaN NaN NaN NaN NaN
13 20 305419896 undefined NaN NaN NaN NaN NaN
13 21 305419896 null 305419896 305419896 0 Infinity NaN
13 22 305419896 str:1 str:3054198961 305419895 305419896 305419896 0
13 23 305419896 str:+1 str:305419896+1 305419895 305419896 305419896 0
13 24 305419896 str:-1.5e1 str:305419896-1.5e1 305419911 -4581298440 -20361326.4 6
13 25 305419896 true 305419897 305419895 305419896 305419896 0
13 26 305419896 0 305419896 305419896 0 Infinity NaN
13 27 305419896 str:1.5e1 str:3054198961.5e1 305419881 4581298440 20361326.4 6
13 28 305419896 str:2e3 str:3054198962e3 305417896 610839792000 152709.948 1896
14 0 34359738367 -Infinity -Infinity Infinity -Infinity -0 34359738367
14 1 34359738367 -1e+100 -1e+100 1e+100 -3.4359738367e+110 -3.4359738367e-90 34359738367
14 2 34359738367 -4294967296 30064771071 38654705663 -147573952585381450000 -7.999999999767169 4294967295
14 3 34359738367 -4294967295 30064771072 38654705662 -147573952551021700000 -8.000000001629815 7
14 4 34359738367 -2147483648 32212254719 36507222015 -73786976292690720000 -15.999999999534339 2147483647
14 5 34359738367 -2147483647 32212254720 36507222014 -73786976258330984000 -16.00000000698492 15
14 6 34359738367 -305419896 34054318471 34665158263 -10494147718636350000 -112.50000021937012 152710015
14 7 34359738367 -1 34359738366 34359738368 -34359738367 -34359738367 0
14 8 34359738367 -1e-100 34359738367 34359738367 -3.4359738367e-90 -3.4359738367e+110 3.887269118643051e-101
14 9 34359738367 -0 34359738367 34359738367 -0 -Infinity NaN
14 10 34359738367 0 34359738367 34359738367 0 Infinity NaN
14 11 34359738367 1e-100 34359738367 34359738367 3.4359738367e-90 3.4359738367e+110 3.887269118643051e-101
14 12 34359738367 1 34359738368 34359738366 34359738367 34359738367 0
14 13 34359738367 305419896 34665158263 34054318471 10494147718636350000 112.50000021937012 152710015
14 14 34359738367 34359738367 68719476734 0 1.1805916206486918e+21 1 0
14 15 34359738367 34359738368 68719476735 -1 1.1805916206830516e+21 0.9999999999708962 34359738367
14 16 34359738367 4294967296 38654705663 30064771071 147573952585381450000 7.999999999767169 4294967295
14 17 34359738367 1e+100 1e+100 -1e+100 3.4359738367e+110 3.4359738367e-90 34359738367
14 18 34359738367 Infinity Infinity -Infinity Infinity 0 34359738367
14 19 34359738367 NaN NaN NaN NaN NaN NaN
14 20 34359738367 undefined NaN NaN NaN NaN NaN
14 21 34359738367 null 34359738367 34359738367 0 Infinity NaN
14 22 34359738367 str:1 str:343597383671 34359738366 34359738367 34359738367 0
14 23 34359738367 str:+1 str:34359738367+1 34359738366 34359738367 34359738367 0
14 24 34359738367 str:-1.5e1 str:34359738367-1.5e1 34359738382 -515396075505 -2290649224.4666667 7
14 25 34359738367 true 34359738368 34359738366 34359738367 34359738367 0
14 26 34359738367 0 34359738367 34359738367 0 Infinity NaN
14 27 34359738367 str:1.5e1 str:343597383671.5e1 34359738352 515396075505 2290649224.4666667 7
14 28 34359738367 str:2e3 str:343597383672e3 34359736367 68719476734000 17179869.1835 367
15 0 34359738368 -Infinity -Infinity Infinity -Infinity -0 34359738368
15 1 34359738368 -1e+100 -1e+100 1e+100 -3.4359738368e+110 -3.4359738368e-90 34359738368
15 2 34359738368 -4294967296 30064771072 38654705664 -147573952589676410000 -8 0
15 3 34359738368 -4294967295 30064771073 38654705663 -147573952555316670000 -8.000000001862645 8
15 4 34359738368 -2147483648 32212254720 36507222016 -73786976294838210000 -16 0
15 5 34359738368 -2147483647 32212254721 36507222015 -73786976260478470000 -16.00000000745058 16
15 6 34359738368 -305419896 34054318472 34665158264 -10494147718941770000 -112.5000002226443 152710016
15 7 34359738368 -1 34359738367 34359738369 -34359738368 -34359738368 0
15 8 34359738368 -1e-100 34359738368 34359738368 -3.4359738368e-90 -3.4359738368e+110 1.9661215833780386e-101
15 9 34359738368 -0 34359738368 34359738368 -0 -Infinity NaN
15 10 34359738368 0 34359738368 34359738368 0 Infinity NaN
15 11 34359738368 1e-100 34359738368 34359738368 3.4359738368e-90 3.4359738368e+110 1.9661215833780386e-101
15 12 34359738368 1 34359738369 34359738367 34359738368 34359738368 0
15 13 34359738368 305419896 34665158264 34054318472 10494147718941770000 112.5000002226443 152710016
15 14 34359738368 34359738367 68719476735 1 1.1805916206830516e+21 1.0000000000291038 1
15 15 34359738368 34359738368 68719476736 0 1.1805916207174113e+21 1 0
15 16 34359738368 4294967296 38654705664 30064771072 147573952589676410000 8 0
15 17 34359738368 1e+100 1e+100 -1e+100 3.4359738368e+110 3.4359738368e-90 34359738368
15 18 34359738368 Infinity Infinity -Infinity Infinity 0 34359738368
15 19 34359738368 NaN NaN NaN NaN NaN NaN
15 20 34359738368 undefined NaN NaN NaN NaN NaN
15 21 34359738368 null 34359738368 34359738368 0 Infinity NaN
15 22 34359738368 str:1 str:343597383681 34359738367 34359738368 34359738368 0
15 23 34359738368 str:+1 str:34359738368+1 34359738367 34359738368 34359738368 0
15 24 34359738368 str:-1.5e1 str:34359738368-1.5e1 34359738383 -515396075520 -2290649224.5333333 8
15 25 34359738368 true 34359738369 34359738367 34359738368 34359738368 0
15 26 34359738368 0 34359738368 34359738368 0 Infinity NaN
15 27 34359738368 str:1.5e1 str:343597383681.5e1 34359738353 515396075520 2290649224.5333333 8
15 28 34359738368 str:2e3 str:343597383682e3 34359736368 68719476736000 17179869.184 368
16 0 4294967296 -Infinity -Infinity Infinity -Infinity -0 4294967296
16 1 4294967296 -1e+100 -1e+100 1e+100 -4.294967296e+109 -4.294967296e-91 4294967296
16 2 4294967296 -4294967296 0 8589934592 -18446744073709552000 -1 0
16 3 4294967296 -4294967295 1 8589934591 -18446744069414584000 -1.0000000002328306 1
16 4 4294967296 -2147483648 2147483648 6442450944 -9223372036854776000 -2 0
16 5 4294967296 -2147483647 2147483649 6442450943 -9223372032559809000 -2.0000000009313226 2
16 6 4294967296 -305419896 3989547400 4600387192 -1311768464867721200 -14.062500027830538 19088752
16 7 4294967296 -1 4294967295 4294967297 -4294967296 -4294967296 0
16 8 4294967296 -1e-100 4294967296 4294967296 -4.294967296e-91 -4.294967296e+109 5.245765197922255e-101
16 9 4294967296 -0 4294967296 4294967296 -0 -Infinity NaN
16 10 4294967296 0 4294967296 4294967296 0 Infinity NaN
16 11 4294967296 1e-100 4294967296 4294967296 4.294967296e-91 4.294967296e+109 5.245765197922255e-101
16 12 4294967296 1 4294967297 4294967295 4294967296 4294967296 0
16 13 4294967296 305419896 4600387192 3989547400 1311768464867721200 14.062500027830538 19088752
16 14 4294967296 34359738367 38654705663 -30064771071 147573952585381450000 0.12500000000363798 4294967296
16 15 4294967296 34359738368 38654705664 -30064771072 147573952589676410000 0.125 4294967296
16 16 4294967296 4294967296 8589934592 0 18446744073709552000 1 0
16 17 4294967296 1e+100 1e+100 -1e+100 4.294967296e+109 4.294967296e-91 4294967296
16 18 4294967296 Infinity Infinity -Infinity Infinity 0 4294967296
16 19 4294967296 NaN NaN NaN NaN NaN NaN
16 20 4294967296 undefined NaN NaN NaN NaN NaN
16 21 4294967296 null 4294967296 4294967296 0 Infinity NaN
16 22 4294967296 str:1 str:42949672961 4294967295 4294967296 4294967296 0
16 23 4294967296 str:+1 str:4294967296+1 4294967295 4294967296 4294967296 0
16 24 4294967296 str:-1.5e1 str:4294967296-1.5e1 4294967311 -64424509440 -286331153.06666666 1
16 25 4294967296 true 4294967297 4294967295 4294967296 4294967296 0
16 26 4294967296 0 4294967296 4294967296 0 Infinity NaN
16 27 4294967296 str:1.5e1 str:42949672961.5e1 4294967281 64424509440 286331153.06666666 1
16 28 4294967296 str:2e3 str:42949672962e3 4294965296 8589934592000 2147483.648 1296
17 0 1e+100 -Infinity -Infinity Infinity -Infinity -0 1e+100
17 1 1e+100 -1e+100 0 2e+100 -1e+200 -1 0
17 2 1e+100 -4294967296 1e+100 1e+100 -4.294967296e+109 -2.3283064365386963e+90 0
17 3 1e+100 -4294967295 1e+100 1e+100 -4.294967295e+109 -2.3283064370807973e+90 707110669
17 4 1e+100 -2147483648 1e+100 1e+100 -2.147483648e+109 -4.656612873077393e+90 0
17 5 1e+100 -2147483647 1e+100 1e+100 -2.147483647e+109 -4.656612875245797e+90 1265806766
17 6 1e+100 -305419896 1e+100 1e+100 -3.05419896e+108 -3.274180932862344e+91 43164928
17 7 1e+100 -1 1e+100 1e+100 -1e+100 -1e+100 0
17 8 1e+100 -1e-100 1e+100 1e+100 -1 -1e+200 9.885752855515831e-101
17 9 1e+100 -0 1e+100 1e+100 -0 -Infinity NaN
17 10 1e+100 0 1e+100 1e+100 0 Infinity NaN
17 11 1e+100 1e-100 1e+100 1e+100 1 1e+200 9.885752855515831e-101
17 12 1e+100 1 1e+100 1e+100 1e+100 1e+100 0
17 13 1e+100 305419896 1e+100 1e+100 3.05419896e+108 3.274180932862344e+91 43164928
17 14 1e+100 34359738367 1e+100 1e+100 3.4359738367e+110 2.910383045758074e+89 22105492658
17 15 1e+100 34359738368 1e+100 1e+100 3.4359738368e+110 2.9103830456733704e+89 0
17 16 1e+100 4294967296 1e+100 1e+100 4.294967296e+109 2.3283064365386963e+90 0
17 17 1e+100 1e+100 2e+100 0 1e+200 1 0
17 18 1e+100 Infinity Infinity -Infinity Infinity 0 1e+100
17 19 1e+100 NaN NaN NaN NaN NaN NaN
17 20 1e+100 undefined NaN NaN NaN NaN NaN
17 21 1e+100 null 1e+100 1e+100 0 Infinity NaN
17 22 1e+100 str:1 str:1e+1001 1e+100 1e+100 1e+100 0
17 23 1e+100 str:+1 str:1e+100+1 1e+100 1e+100 1e+100 0
17 24 1e+100 str:-1.5e1 str:1e+100-1.5e1 1e+100 -1.5e+101 -6.666666666666666e+98 4
17 25 1e+100 true 1e+100 1e+100 1e+100 1e+100 0
17 26 1e+100 0 1e+100 1e+100 0 Infinity NaN
17 27 1e+100 str:1.5e1 str:1e+1001.5e1 1e+100 1.5e+101 6.666666666666666e+98 4
17 28 1e+100 str:2e3 str:1e+1002e3 1e+100 2e+103 5e+96 1104
18 0 Infinity -Infinity NaN Infinity -Infinity NaN NaN
18 1 Infinity -1e+100 Infinity Infinity -Infinity -Infinity NaN
18 2 Infinity -4294967296 Infinity Infinity -Infinity -Infinity NaN
18 3 Infinity -4294967295 Infinity Infinity -Infinity -Infinity NaN
18 4 Infinity -2147483648 Infinity Infinity -Infinity -Infinity NaN
18 5 Infinity -2147483647 Infinity Infinity -Infinity -Infinity NaN
18 6 Infinity -305419896 Infinity Infinity -Infinity -Infinity NaN
18 7 Infinity -1 Infinity Infinity -Infinity -Infinity NaN
18 8 Infinity -1e-100 Infinity Infinity -Infinity -Infinity NaN
18 9 Infinity -0 Infinity Infinity NaN -Infinity NaN
18 10 Infinity 0 Infinity Infinity NaN Infinity NaN
18 11 Infinity 1e-100 Infinity Infinity Infinity Infinity NaN
18 12 Infinity 1 Infinity Infinity Infinity Infinity NaN
18 13 Infinity 305419896 Infinity Infinity Infinity Infinity NaN
18 14 Infinity 34359738367 Infinity Infinity Infinity Infinity NaN
18 15 Infinity 34359738368 Infinity Infinity Infinity Infinity NaN
18 16 Infinity 4294967296 Infinity Infinity Infinity Infinity NaN
18 17 Infinity 1e+100 Infinity Infinity Infinity Infinity NaN
18 18 Infinity Infinity Infinity NaN Infinity NaN NaN
18 19 Infinity NaN NaN NaN NaN NaN NaN
18 20 Infinity undefined NaN NaN NaN NaN NaN
18 21 Infinity null Infinity Infinity NaN Infinity NaN
18 22 Infinity str:1 str:Infinity1 Infinity Infinity Infinity NaN
18 23 Infinity str:+1 str:Infinity+1 Infinity Infinity Infinity NaN
18 24 Infinity str:-1.5e1 str:Infinity-1.5e1 Infinity -Infinity -Infinity NaN
18 25 Infinity true Infinity Infinity Infinity Infinity NaN
18 26 Infinity 0 Infinity Infinity NaN Infinity NaN
18 27 Infinity str:1.5e1 str:Infinity1.5e1 Infinity Infinity Infinity NaN
18 28 Infinity str:2e3 str:Infinity2e3 Infinity Infinity Infinity NaN
19 0 NaN -Infinity NaN NaN NaN NaN NaN
19 1 NaN -1e+100 NaN NaN NaN NaN NaN
19 2 NaN -4294967296 NaN NaN NaN NaN NaN
19 3 NaN -4294967295 NaN NaN NaN NaN NaN
19 4 NaN -2147483648 NaN NaN NaN NaN NaN
19 5 NaN -2147483647 NaN NaN NaN NaN NaN
19 6 NaN -305419896 NaN NaN NaN NaN NaN
19 7 NaN -1 NaN NaN NaN NaN NaN
19 8 NaN -1e-100 NaN NaN NaN NaN NaN
19 9 NaN -0 NaN NaN NaN NaN NaN
19 10 NaN 0 NaN NaN NaN NaN NaN
19 11 NaN 1e-100 NaN NaN NaN NaN NaN
19 12 NaN 1 NaN NaN NaN NaN NaN
19 13 NaN 305419896 NaN NaN NaN NaN NaN
19 14 NaN 34359738367 NaN NaN NaN NaN NaN
19 15 NaN 34359738368 NaN NaN NaN NaN NaN
19 16 NaN 4294967296 NaN NaN NaN NaN NaN
19 17 NaN 1e+100 NaN NaN NaN NaN NaN
19 18 NaN Infinity NaN NaN NaN NaN NaN
19 19 NaN NaN NaN NaN NaN NaN NaN
19 20 NaN undefined NaN NaN NaN NaN NaN
19 21 NaN null NaN NaN NaN NaN NaN
19 22 NaN str:1 str:NaN1 NaN NaN NaN NaN
19 23 NaN str:+1 str:NaN+1 NaN NaN NaN NaN
19 24 NaN str:-1.5e1 str:NaN-1.5e1 NaN NaN NaN NaN
19 25 NaN true NaN NaN NaN NaN NaN
19 26 NaN 0 NaN NaN NaN NaN NaN
19 27 NaN str:1.5e1 str:NaN1.5e1 NaN NaN NaN NaN
19 28 NaN str:2e3 str:NaN2e3 NaN NaN NaN NaN
20 0 undefined -Infinity NaN NaN NaN NaN NaN
20 1 undefined -1e+100 NaN NaN NaN NaN NaN
20 2 undefined -4294967296 NaN NaN NaN NaN NaN
20 3 undefined -4294967295 NaN NaN NaN NaN NaN
20 4 undefined -2147483648 NaN NaN NaN NaN NaN
20 5 undefined -2147483647 NaN NaN NaN NaN NaN
20 6 undefined -305419896 NaN NaN NaN NaN NaN
20 7 undefined -1 NaN NaN NaN NaN NaN
20 8 undefined -1e-100 NaN NaN NaN NaN NaN
20 9 undefined -0 NaN NaN NaN NaN NaN
20 10 undefined 0 NaN NaN NaN NaN NaN
20 11 undefined 1e-100 NaN NaN NaN NaN NaN
20 12 undefined 1 NaN NaN NaN NaN NaN
20 13 undefined 305419896 NaN NaN NaN NaN NaN
20 14 undefined 34359738367 NaN NaN NaN NaN NaN
20 15 undefined 34359738368 NaN NaN NaN NaN NaN
20 16 undefined 4294967296 NaN NaN NaN NaN NaN
20 17 undefined 1e+100 NaN NaN NaN NaN NaN
20 18 undefined Infinity NaN NaN NaN NaN NaN
20 19 undefined NaN NaN NaN NaN NaN NaN
20 20 undefined undefined NaN NaN NaN NaN NaN
20 21 undefined null NaN NaN NaN NaN NaN
20 22 undefined str:1 str:undefined1 NaN NaN NaN NaN
20 23 undefined str:+1 str:undefined+1 NaN NaN NaN NaN
20 24 undefined str:-1.5e1 str:undefined-1.5e1 NaN NaN NaN NaN
20 25 undefined true NaN NaN NaN NaN NaN
20 26 undefined 0 NaN NaN NaN NaN NaN
20 27 undefined str:1.5e1 str:undefined1.5e1 NaN NaN NaN NaN
20 28 undefined str:2e3 str:undefined2e3 NaN NaN NaN NaN
21 0 null -Infinity -Infinity Infinity NaN -0 0
21 1 null -1e+100 -1e+100 1e+100 -0 -0 0
21 2 null -4294967296 -4294967296 4294967296 -0 -0 0
21 3 null -4294967295 -4294967295 4294967295 -0 -0 0
21 4 null -2147483648 -2147483648 2147483648 -0 -0 0
21 5 null -2147483647 -2147483647 2147483647 -0 -0 0
21 6 null -305419896 -305419896 305419896 -0 -0 0
21 7 null -1 -1 1 -0 -0 0
21 8 null -1e-100 -1e-100 1e-100 -0 -0 0
21 9 null -0 0 0 -0 NaN NaN
21 10 null 0 0 0 0 NaN NaN
21 11 null 1e-100 1e-100 -1e-100 0 0 0
21 12 null 1 1 -1 0 0 0
21 13 null 305419896 305419896 -305419896 0 0 0
21 14 null 34359738367 34359738367 -34359738367 0 0 0
21 15 null 34359738368 34359738368 -34359738368 0 0 0
21 16 null 4294967296 4294967296 -4294967296 0 0 0
21 17 null 1e+100 1e+100 -1e+100 0 0 0
21 18 null Infinity Infinity -Infinity NaN 0 0
21 19 null NaN NaN NaN NaN NaN NaN
21 20 null undefined NaN NaN NaN NaN NaN
21 21 null null 0 0 0 NaN NaN
21 22 null str:1 str:null1 -1 0 0 0
21 23 null str:+1 str:null+1 -1 0 0 0
21 24 null str:-1.5e1 str:null-1.5e1 15 -0 -0 0
21 25 null true 1 -1 0 0 0
21 26 null 0 0 0 0 NaN NaN
21 27 null str:1.5e1 str:null1.5e1 -15 0 0 0
21 28 null str:2e3 str:null2e3 -2000 0 0 0
22 0 str:1 -Infinity str:1-Infinity Infinity -Infinity -0 1
22 1 str:1 -1e+100 str:1-1e+100 1e+100 -1e+100 -1e-100 1
22 2 str:1 -4294967296 str:1-4294967296 4294967297 -4294967296 -2.3283064365386963e-10 1
22 3 str:1 -4294967295 str:1-4294967295 4294967296 -4294967295 -2.3283064370807974e-10 1
22 4 str:1 -2147483648 str:1-2147483648 2147483649 -2147483648 -4.656612873077393e-10 1
22 5 str:1 -2147483647 str:1-2147483647 2147483648 -2147483647 -4.656612875245797e-10 1
22 6 str:1 -305419896 str:1-305419896 305419897 -305419896 -3.2741809328623437e-9 1
22 7 str:1 -1 str:1-1 2 -1 -1 0
22 8 str:1 -1e-100 str:1-1e-100 1 -1e-100 -1e+100 8.078852464734988e-101
22 9 str:1 -0 str:10 1 -0 -Infinity NaN
22 10 str:1 0 str:10 1 0 Infinity NaN
22 11 str:1 1e-100 str:11e-100 1 1e-100 1e+100 8.078852464734988e-101
22 12 str:1 1 str:11 0 1 1 0
22 13 str:1 305419896 str:1305419896 -305419895 305419896 3.2741809328623437e-9 1
22 14 str:1 34359738367 str:134359738367 -34359738366 34359738367 2.9103830457580737e-11 1
22 15 str:1 34359738368 str:134359738368 -34359738367 34359738368 2.9103830456733704e-11 1
22 16 str:1 4294967296 str:14294967296 -4294967295 4294967296 2.3283064365386963e-10 1
22 17 str:1 1e+100 str:11e+100 -1e+100 1e+100 1e-100 1
22 18 str:1 Infinity str:1Infinity -Infinity Infinity 0 1
22 19 str:1 NaN str:1NaN NaN NaN NaN NaN
22 20 str:1 undefined str:1undefined NaN NaN NaN NaN
22 21 str:1 null str:1null 1 0 Infinity NaN
22 22 str:1 str:1 str:11 0 1 1 0
22 23 str:1 str:+1 str:1+1 0 1 1 0
22 24 str:1 str:-1.5e1 str:1-1.5e1 16 -15 -0.06666666666666667 1
22 25 str:1 true str:1true 0 1 1 0
22 26 str:1 0 str:1false 1 0 Infinity NaN
22 27 str:1 str:1.5e1 str:11.5e1 -14 15 0.06666666666666667 1
22 28 str:1 str:2e3 str:12e3 -1999 2000 0.0005 1
23 0 str:+1 -Infinity str:+1-Infinity Infinity -Infinity -0 1
23 1 str:+1 -1e+100 str:+1-1e+100 1e+100 -1e+100 -1e-100 1
23 2 str:+1 -4294967296 str:+1-4294967296 4294967297 -4294967296 -2.3283064365386963e-10 1
23 3 str:+1 -4294967295 str:+1-4294967295 4294967296 -4294967295 -2.3283064370807974e-10 1
23 4 str:+1 -2147483648 str:+1-2147483648 2147483649 -2147483648 -4.656612873077393e-10 1
23 5 str:+1 -2147483647 str:+1-2147483647 2147483648 -2147483647 -4.656612875245797e-10 1
23 6 str:+1 -305419896 str:+1-305419896 305419897 -305419896 -3.2741809328623437e-9 1
23 7 str:+1 -1 str:+1-1 2 -1 -1 0
23 8 str:+1 -1e-100 str:+1-1e-100 1 -1e-100 -1e+100 8.078852464734988e-101
23 9 str:+1 -0 str:+10 1 -0 -Infinity NaN
23 10 str:+1 0 str:+10 1 0 Infinity NaN
23 11 str:+1 1e-100 str:+11e-100 1 1e-100 1e+100 8.078852464734988e-101
23 12 str:+1 1 str:+11 0 1 1 0
23 13 str:+1 305419896 str:+1305419896 -305419895 305419896 3.2741809328623437e-9 1
23 14 str:+1 34359738367 str:+134359738367 -34359738366 34359738367 2.9103830457580737e-11 1
23 15 str:+1 34359738368 str:+134359738368 -34359738367 34359738368 2.9103830456733704e-11 1
23 16 str:+1 4294967296 str:+14294967296 -4294967295 4294967296 2.3283064365386963e-10 1
23 17 str:+1 1e+100 str:+11e+100 -1e+100 1e+100 1e-100 1
23 18 str:+1 Infinity str:+1Infinity -Infinity Infinity 0 1
23 19 str:+1 NaN str:+1NaN NaN NaN NaN NaN
23 20 str:+1 undefined str:+1undefined NaN NaN NaN NaN
23 21 str:+1 null str:+1null 1 0 Infinity NaN
23 22 str:+1 str:1 str:+11 0 1 1 0
23 23 str:+1 str:+1 str:+1+1 0 1 1 0
23 24 str:+1 str:-1.5e1 str:+1-1.5e1 16 -15 -0.06666666666666667 1
23 25 str:+1 true str:+1true 0 1 1 0
23 26 str:+1 0 str:+1false 1 0 Infinity NaN
23 27 str:+1 str:1.5e1 str:+11.5e1 -14 15 0.06666666666666667 1
23 28 str:+1 str:2e3 str:+12e3 -1999 2000 0.0005 1
24 0 str:-1.5e1 -Infinity str:-1.5e1-Infinity Infinity Infinity 0 -15
24 1 str:-1.5e1 -1e+100 str:-1.5e1-1e+100 1e+100 1.5e+101 1.5e-99 -15
24 2 str:-1.5e1 -4294967296 str:-1.5e1-4294967296 4294967281 64424509440 3.4924596548080444e-9 -15
24 3 str:-1.5e1 -4294967295 str:-1.5e1-4294967295 4294967280 64424509425 3.492459655621196e-9 -15
24 4 str:-1.5e1 -2147483648 str:-1.5e1-2147483648 2147483633 32212254720 6.984919309616089e-9 -15
24 5 str:-1.5e1 -2147483647 str:-1.5e1-2147483647 2147483632 32212254705 6.984919312868695e-9 -15
24 6 str:-1.5e1 -305419896 str:-1.5e1-305419896 305419881 4581298440 4.9112713992935156e-8 -15
24 7 str:-1.5e1 -1 str:-1.5e1-1 -14 15 15 -0
24 8 str:-1.5e1 -1e-100 str:-1.5e1-1e-100 -15 1.5e-99 1.5e+101 -1.1827869710248145e-101
24 9 str:-1.5e1 -0 str:-1.5e10 -15 0 Infinity NaN
24 10 str:-1.5e1 0 str:-1.5e10 -15 -0 -Infinity NaN
24 11 str:-1.5e1 1e-100 str:-1.5e11e-100 -15 -1.5e-99 -1.5e+101 -1.1827869710248145e-101
24 12 str:-1.5e1 1 str:-1.5e11 -16 -15 -15 -0
24 13 str:-1.5e1 305419896 str:-1.5e1305419896 -305419911 -4581298440 -4.9112713992935156e-8 -15
24 14 str:-1.5e1 34359738367 str:-1.5e134359738367 -34359738382 -515396075505 -4.3655745686371105e-10 -15
24 15 str:-1.5e1 34359738368 str:-1.5e134359738368 -34359738383 -515396075520 -4.3655745685100555e-10 -15
24 16 str:-1.5e1 4294967296 str:-1.5e14294967296 -4294967311 -64424509440 -3.4924596548080444e-9 -15
24 17 str:-1.5e1 1e+100 str:-1.5e11e+100 -1e+100 -1.5e+101 -1.5e-99 -15
24 18 str:-1.5e1 Infinity str:-1.5e1Infinity -Infinity -Infinity -0 -15
24 19 str:-1.5e1 NaN str:-1.5e1NaN NaN NaN NaN NaN
24 20 str:-1.5e1 undefined str:-1.5e1undefined NaN NaN NaN NaN
24 21 str:-1.5e1 null str:-1.5e1null -15 -0 -Infinity NaN
24 22 str:-1.5e1 str:1 str:-1.5e11 -16 -15 -15 -0
24 23 str:-1.5e1 str:+1 str:-1.5e1+1 -16 -15 -15 -0
24 24 str:-1.5e1 str:-1.5e1 str:-1.5e1-1.5e1 0 225 1 -0
24 25 str:-1.5e1 true str:-1.5e1true -16 -15 -15 -0
24 26 str:-1.5e1 0 str:-1.5e1false -15 -0 -Infinity NaN
24 27 str:-1.5e1 str:1.5e1 str:-1.5e11.5e1 -30 -225 -1 -0
24 28 str:-1.5e1 str:2e3 str:-1.5e12e3 -2015 -30000 -0.0075 -15
25 0 true -Infinity -Infinity Infinity -Infinity -0 1
25 1 true -1e+100 -1e+100 1e+100 -1e+100 -1e-100 1
25 2 true -4294967296 -4294967295 4294967297 -4294967296 -2.3283064365386963e-10 1
25 3 true -4294967295 -4294967294 4294967296 -4294967295 -2.3283064370807974e-10 1
25 4 true -2147483648 -2147483647 2147483649 -2147483648 -4.656612873077393e-10 1
25 5 true -2147483647 -2147483646 2147483648 -2147483647 -4.656612875245797e-10 1
25 6 true -305419896 -305419895 305419897 -305419896 -3.2741809328623437e-9 1
25 7 true -1 0 2 -1 -1 0
25 8 true -1e-100 1 1 -1e-100 -1e+100 8.078852464734988e-101
25 9 true -0 1 1 -0 -Infinity NaN
25 10 true 0 1 1 0 Infinity NaN
25 11 true 1e-100 1 1 1e-100 1e+100 8.078852464734988e-101
25 12 true 1 2 0 1 1 0
25 13 true 305419896 305419897 -305419895 305419896 3.2741809328623437e-9 1
25 14 true 34359738367 34359738368 -34359738366 34359738367 2.9103830457580737e-11 1
25 15 true 34359738368 34359738369 -34359738367 34359738368 2.9103830456733704e-11 1
25 16 true 4294967296 4294967297 -4294967295 4294967296 2.3283064365386963e-10 1
25 17 true 1e+100 1e+100 -1e+100 1e+100 1e-100 1
25 18 true Infinity Infinity -Infinity Infinity 0 1
25 19 true NaN NaN NaN NaN NaN NaN
25 20 true undefined NaN NaN NaN NaN NaN
25 21 true null 1 1 0 Infinity NaN
25 22 true str:1 str:true1 0 1 1 0
25 23 true str:+1 str:true+1 0 1 1 0
25 24 true str:-1.5e1 str:true-1.5e1 16 -15 -0.06666666666666667 1
25 25 true true 2 0 1 1 0
25 26 true 0 1 1 0 Infinity NaN
25 27 true str:1.5e1 str:true1.5e1 -14 15 0.06666666666666667 1
25 28 true str:2e3 str:true2e3 -1999 2000 0.0005 1
26 0 0 -Infinity -Infinity Infinity NaN -0 0
26 1 0 -1e+100 -1e+100 1e+100 -0 -0 0
26 2 0 -4294967296 -4294967296 4294967296 -0 -0 0
26 3 0 -4294967295 -4294967295 4294967295 -0 -0 0
26 4 0 -2147483648 -2147483648 2147483648 -0 -0 0
26 5 0 -2147483647 -2147483647 2147483647 -0 -0 0
26 6 0 -305419896 -305419896 305419896 -0 -0 0
26 7 0 -1 -1 1 -0 -0 0
26 8 0 -1e-100 -1e-100 1e-100 -0 -0 0
26 9 0 -0 0 0 -0 NaN NaN
26 10 0 0 0 0 0 NaN NaN
26 11 0 1e-100 1e-100 -1e-100 0 0 0
26 12 0 1 1 -1 0 0 0
26 13 0 305419896 305419896 -305419896 0 0 0
26 14 0 34359738367 34359738367 -34359738367 0 0 0
26 15 0 34359738368 34359738368 -34359738368 0 0 0
26 16 0 4294967296 4294967296 -4294967296 0 0 0
26 17 0 1e+100 1e+100 -1e+100 0 0 0
26 18 0 Infinity Infinity -Infinity NaN 0 0
26 19 0 NaN NaN NaN NaN NaN NaN
26 20 0 undefined NaN NaN NaN NaN NaN
26 21 0 null 0 0 0 NaN NaN
26 22 0 str:1 str:false1 -1 0 0 0
26 23 0 str:+1 str:false+1 -1 0 0 0
26 24 0 str:-1.5e1 str:false-1.5e1 15 -0 -0 0
26 25 0 true 1 -1 0 0 0
26 26 0 0 0 0 0 NaN NaN
26 27 0 str:1.5e1 str:false1.5e1 -15 0 0 0
26 28 0 str:2e3 str:false2e3 -2000 0 0 0
27 0 str:1.5e1 -Infinity str:1.5e1-Infinity Infinity -Infinity -0 15
27 1 str:1.5e1 -1e+100 str:1.5e1-1e+100 1e+100 -1.5e+101 -1.5e-99 15
27 2 str:1.5e1 -4294967296 str:1.5e1-4294967296 4294967311 -64424509440 -3.4924596548080444e-9 15
27 3 str:1.5e1 -4294967295 str:1.5e1-4294967295 4294967310 -64424509425 -3.492459655621196e-9 15
27 4 str:1.5e1 -2147483648 str:1.5e1-2147483648 2147483663 -32212254720 -6.984919309616089e-9 15
27 5 str:1.5e1 -2147483647 str:1.5e1-2147483647 2147483662 -32212254705 -6.984919312868695e-9 15
27 6 str:1.5e1 -305419896 str:1.5e1-305419896 305419911 -4581298440 -4.9112713992935156e-8 15
27 7 str:1.5e1 -1 str:1.5e1-1 16 -15 -15 0
27 8 str:1.5e1 -1e-100 str:1.5e1-1e-100 15 -1.5e-99 -1.5e+101 1.1827869710248145e-101
27 9 str:1.5e1 -0 str:1.5e10 15 -0 -Infinity NaN
27 10 str:1.5e1 0 str:1.5e10 15 0 Infinity NaN
27 11 str:1.5e1 1e-100 str:1.5e11e-100 15 1.5e-99 1.5e+101 1.1827869710248145e-101
27 12 str:1.5e1 1 str:1.5e11 14 15 15 0
27 13 str:1.5e1 305419896 str:1.5e1305419896 -305419881 4581298440 4.9112713992935156e-8 15
27 14 str:1.5e1 34359738367 str:1.5e134359738367 -34359738352 515396075505 4.3655745686371105e-10 15
27 15 str:1.5e1 34359738368 str:1.5e134359738368 -34359738353 515396075520 4.3655745685100555e-10 15
27 16 str:1.5e1 4294967296 str:1.5e14294967296 -4294967281 64424509440 3.4924596548080444e-9 15
27 17 str:1.5e1 1e+100 str:1.5e11e+100 -1e+100 1.5e+101 1.5e-99 15
27 18 str:1.5e1 Infinity str:1.5e1Infinity -Infinity Infinity 0 15
27 19 str:1.5e1 NaN str:1.5e1NaN NaN NaN NaN NaN
27 20 str:1.5e1 undefined str:1.5e1undefined NaN NaN NaN NaN
27 21 str:1.5e1 null str:1.5e1null 15 0 Infinity NaN
27 22 str:1.5e1 str:1 str:1.5e11 14 15 15 0
27 23 str:1.5e1 str:+1 str:1.5e1+1 14 15 15 0
27 24 str:1.5e1 str:-1.5e1 str:1.5e1-1.5e1 30 -225 -1 0
27 25 str:1.5e1 true str:1.5e1true 14 15 15 0
27 26 str:1.5e1 0 str:1.5e1false 15 0 Infinity NaN
27 27 str:1.5e1 str:1.5e1 str:1.5e11.5e1 0 225 1 0
27 28 str:1.5e1 str:2e3 str:1.5e12e3 -1985 30000 0.0075 15
28 0 str:2e3 -Infinity str:2e3-Infinity Infinity -Infinity -0 2000
28 1 str:2e3 -1e+100 str:2e3-1e+100 1e+100 -2e+103 -2e-97 2000
28 2 str:2e3 -4294967296 str:2e3-4294967296 4294969296 -8589934592000 -4.6566128730773926e-7 2000
28 3 str:2e3 -4294967295 str:2e3-4294967295 4294969295 -8589934590000 -4.656612874161595e-7 2000
28 4 str:2e3 -2147483648 str:2e3-2147483648 2147485648 -4294967296000 -9.313225746154785e-7 2000
28 5 str:2e3 -2147483647 str:2e3-2147483647 2147485647 -4294967294000 -9.313225750491594e-7 2000
28 6 str:2e3 -305419896 str:2e3-305419896 305421896 -610839792000 -0.000006548361865724687 2000
28 7 str:2e3 -1 str:2e3-1 2001 -2000 -2000 0
28 8 str:2e3 -1e-100 str:2e3-1e-100 2000 -2e-97 -2e+103 7.70492946997527e-101
28 9 str:2e3 -0 str:2e30 2000 -0 -Infinity NaN
28 10 str:2e3 0 str:2e30 2000 0 Infinity NaN
28 11 str:2e3 1e-100 str:2e31e-100 2000 2e-97 2e+103 7.70492946997527e-101
28 12 str:2e3 1 str:2e31 1999 2000 2000 0
28 13 str:2e3 305419896 str:2e3305419896 -305417896 610839792000 0.000006548361865724687 2000
28 14 str:2e3 34359738367 str:2e334359738367 -34359736367 68719476734000 5.820766091516147e-8 2000
28 15 str:2e3 34359738368 str:2e334359738368 -34359736368 68719476736000 5.820766091346741e-8 2000
28 16 str:2e3 4294967296 str:2e34294967296 -4294965296 8589934592000 4.6566128730773926e-7 2000
28 17 str:2e3 1e+100 str:2e31e+100 -1e+100 2e+103 2e-97 2000
28 18 str:2e3 Infinity str:2e3Infinity -Infinity Infinity 0 2000
28 19 str:2e3 NaN str:2e3NaN NaN NaN NaN NaN
28 20 str:2e3 undefined str:2e3undefined NaN NaN NaN NaN
28 21 str:2e3 null str:2e3null 2000 0 Infinity NaN
28 22 str:2e3 str:1 str:2e31 1999 2000 2000 0
28 23 str:2e3 str:+1 str:2e3+1 1999 2000 2000 0
28 24 str:2e3 str:-1.5e1 str:2e3-1.5e1 2015 -30000 -133.33333333333334 5
28 25 str:2e3 true str:2e3true 1999 2000 2000 0
28 26 str:2e3 0 str:2e3false 2000 0 Infinity NaN
28 27 str:2e3 str:1.5e1 str:2e31.5e1 1985 30000 133.33333333333334 5
28 28 str:2e3 str:2e3 str:2e32e3 0 4000000 1 0
0 -Infinity -Infinity Infinity
1 -1e+100 -1e+100 1e+100
2 -4294967296 -4294967296 4294967296
3 -4294967295 -4294967295 4294967295
4 -2147483648 -2147483648 2147483648
5 -2147483647 -2147483647 2147483647
6 -305419896 -305419896 305419896
7 -1 -1 1
8 -1e-100 -1e-100 1e-100
9 -0 -0 0
10 0 0 -0
11 1e-100 1e-100 -1e-100
12 1 1 -1
13 305419896 305419896 -305419896
14 34359738367 34359738367 -34359738367
15 34359738368 34359738368 -34359738368
16 4294967296 4294967296 -4294967296
17 1e+100 1e+100 -1e+100
18 Infinity Infinity -Infinity
19 NaN NaN NaN
20 undefined NaN NaN
21 null 0 -0
22 str:1 1 -1
23 str:+1 1 -1
24 str:-1.5e1 -15 15
25 true 1 -1
26 0 0 -0
27 str:1.5e1 15 -15
28 str:2e3 2000 -2000
===*/

function matrixTest() {
    var vals = testValues;
    var i, j;

    for (i = 0; i < vals.length; i++) {
        for (j = 0; j < vals.length; j++) {
            print(i, j, fmt(vals[i]), fmt(vals[j]),
                  fmt(vals[i] + vals[j]),
                  fmt(vals[i] - vals[j]),
                  fmt(vals[i] * vals[j]),
                  fmt(vals[i] / vals[j]),
                  fmt(vals[i] % vals[j]));
        }
    }

    for (i = 0; i < vals.length; i++) {
        print(i, fmt(vals[i]),
              fmt(+vals[i]), fmt(-vals[i]));
    }
}

try {
    print('matrix test');
    matrixTest();
} catch (e) {
    print(e.stack || e);
}


/*===
multiply rounding to zero and infinity
Infinity number
Infinity number
Infinity number
0 number Infinity
===*/

function multiplyRoundingTest() {
    var t;

    t = 1e200 * 1e200;
    print (t, typeof t);

    t = 1e200;
    t *= 1e200;
    t *= 1e-200;
    t *= 1e-200;
    print (t, typeof t);

    // same thing as above, '*' is left associative
    t = 1e200 * 1e200 * 1e-200 * 1e-200;
    print (t, typeof t);

    // (1e200 * 1e-200) == 1, then goes to zero
    t = 1e200 * 1e-200 * 1e-200 * 1e-200;
    print (t, typeof t, 1 / t);
}

try {
    print('multiply rounding to zero and infinity');
    multiplyRoundingTest();
} catch (e) {
    print(e.stack || e);
}
