/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     macro.c                                                        */
/*                                                                          */
/*                                                                          */
/* description:  dimension dependent part of reading/writing macro          */
/*               triangulations for 3d                                      */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Albert-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg                                             */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* new_bound_val_3d(): determine the value of the subsimplex being checked  */
/* The type of a boundary subsimplex is set to the highest type of all      */
/* boundary faces containg the subsimplex.                                  */
/*--------------------------------------------------------------------------*/

static S_CHAR new_bound_val_3d(S_CHAR face, S_CHAR edge_or_vertex)
{
  if (face) {
    if (face >= DIRICHLET)
      return(MAX(face, edge_or_vertex));
    else
      if (edge_or_vertex != INTERIOR)
       	return(MAX(face, edge_or_vertex));
      else
	return(face);
  }

  return(edge_or_vertex);
}


/*--------------------------------------------------------------------------*/
/* fill_bound_info_3d(): fill boundary info on faces from MACRO_DATA *data  */
/* into corresponding macro elements, set boundary types of vertices (edges */
/* are handled later in fill_more_bound_info_3d) using new_bound_val_3d()   */
/*--------------------------------------------------------------------------*/

static void fill_bound_info_3d(MESH *mesh, const MACRO_DATA *data)
{
  MACRO_EL   *mel = mesh->macro_els;
  int         i, j, k;
  int         ne = mesh->n_elements;
  int         nv = mesh->n_vertices;

  S_CHAR  *bound = MEM_ALLOC(nv, S_CHAR);

/*--------------------------------------------------------------------------*/
/* The first N_FACES_3D entries of mel[].boundary will contain pointers to  */
/* BOUNDARY structures corresponding to the elements' faces. These are the  */
/* crucial infos provided externally, the other subsimplices (here vertices)*/
/* are set to a boundary type depending on the face containing them.        */
/*--------------------------------------------------------------------------*/

  for(i = 0; i < data->n_macro_elements; i++)
    for(j = 0; j < N_FACES_3D; j++)
      mel[i].face_bound[j] = data->boundary[NEIGH_IND(3,i,j)];

  for (i = 0; i < nv; i++)
    bound[i] = INTERIOR;

  for (i = 0; i < ne; i++) {
    for (j = 0; j < N_FACES_3D; j++) {
       for (k = 1; k < 4; k++)
         bound[data->mel_vertices[VERT_IND(3,i,(j+k)%4)]] =
	   new_bound_val_3d(mel[i].face_bound[j],
			    bound[data->mel_vertices[VERT_IND(3,i,(j+k)%4)]]);
    }
  }

  for (i = 0; i < ne; i++)
    for (j = 0; j < N_VERTICES_3D; j++)
      mel[i].vertex_bound[j] = bound[data->mel_vertices[VERT_IND(3,i,j)]];

  MEM_FREE(bound, nv, S_CHAR);

  return;
}


/*--------------------------------------------------------------------------*/
/* new_edge_3d(): This procedure counts the number of faces                 */
/* sharing the edge. Another important task is setting the boundary type of */
/* the edge depending on the types of these faces via new_bound_val_3d().   */
/*--------------------------------------------------------------------------*/

static int new_edge_3d(const MACRO_DATA *data,
		       MACRO_EL *mel, int mel_edge_no,int *n_neigh)
{
  FUNCNAME("new_edge_3d"); 

  typedef S_CHAR *LIST_BOUND;

  MACRO_EL       *neigh;
  int             j, k, opp_v, mel_index, edge_no, vertex[2];
  S_CHAR          bound = INTERIOR;
  int             max_no_list_el = 20;
  LIST_BOUND     *list_bound = MEM_ALLOC(max_no_list_el, LIST_BOUND);
  static int      next_el[6][2] = {{3,2},{1,3},{1,2},{0,3},{0,2},{0,1}};


  mel_index = mel->index;
  edge_no = mel_edge_no;

  list_bound[0] = &(mel->edge_bound[edge_no]);

  for (j = 0; j < 2; j++)
    vertex[j] = data->mel_vertices
      [VERT_IND(3,mel_index,vertex_of_edge_3d[edge_no][j])];
  
/*--------------------------------------------------------------------------*/
/*  first look for all neighbours in one direction until a boundary is      */
/*  reached :-( or we are back to mel :-)                                   */
/*  if the index of a neighbour is smaller than the element index, the edge */
/*  is counted by this neighbour, return 0.                                 */
/*  If we are back to element, return 1, to count the edge                  */
/*--------------------------------------------------------------------------*/

  neigh = mel->neigh[next_el[edge_no][0]];
  opp_v = mel->opp_vertex[next_el[edge_no][0]];

  bound = new_bound_val_3d(mel->face_bound[next_el[edge_no][0]], bound);

  while (neigh  &&  neigh != mel) {
    if (neigh->index < mel_index) {
      MEM_FREE(list_bound, max_no_list_el, LIST_BOUND);
      return(0);
    }
    
    for (j = 0; j < N_VERTICES_3D; j++)
      if (data->mel_vertices[VERT_IND(3,neigh->index,j)] == vertex[0]) break;
    for (k = 0; k < N_VERTICES_3D; k++)
      if (data->mel_vertices[VERT_IND(3,neigh->index,k)] == vertex[1]) break;
    edge_no = edge_of_dofs_3d[j][k];

    if (*n_neigh == max_no_list_el) {
      list_bound = MEM_REALLOC(list_bound, max_no_list_el, max_no_list_el + 20,
                               LIST_BOUND);
      max_no_list_el += 20;
    } 

    list_bound[(*n_neigh)++] = &(neigh->edge_bound[edge_no]);

    if (next_el[edge_no][0] != opp_v) {
      bound = new_bound_val_3d(neigh->face_bound[next_el[edge_no][0]], bound);

      opp_v = neigh->opp_vertex[next_el[edge_no][0]];
      neigh = neigh->neigh[next_el[edge_no][0]];
    }
    else {
      bound = new_bound_val_3d(neigh->face_bound[next_el[edge_no][1]], bound);

      opp_v = neigh->opp_vertex[next_el[edge_no][1]];
      neigh = neigh->neigh[next_el[edge_no][1]];
    }
  }

  if (!neigh) {
/*--------------------------------------------------------------------------*/
/*  while looping around the edge the domain's boundary was reached. Now,   */
/*  loop in the other direction to the domain's boundary		    */
/*--------------------------------------------------------------------------*/
    edge_no = mel_edge_no;

    neigh = mel->neigh[next_el[edge_no][1]];
    opp_v = mel->opp_vertex[next_el[edge_no][1]];

    bound = new_bound_val_3d(mel->face_bound[next_el[edge_no][1]], bound);

    while (neigh) {
      if (neigh->index < mel_index) {
        MEM_FREE(list_bound, max_no_list_el, LIST_BOUND);
        return(0);
      }

      for (j = 0; j < N_VERTICES_3D; j++)
	if (data->mel_vertices[VERT_IND(3,neigh->index,j)] == vertex[0]) break;
      for (k = 0; k < N_VERTICES_3D; k++)
	if (data->mel_vertices[VERT_IND(3,neigh->index,k)] == vertex[1]) break;
      edge_no = edge_of_dofs_3d[j][k];

      if (*n_neigh == max_no_list_el) {
        list_bound = MEM_REALLOC(list_bound, max_no_list_el,
                                 max_no_list_el + 20, LIST_BOUND);
        max_no_list_el += 20;
      } 

      list_bound[(*n_neigh)++] = &(neigh->edge_bound[edge_no]);

      if (next_el[edge_no][0] != opp_v) {
	bound = new_bound_val_3d(neigh->face_bound[next_el[edge_no][0]],bound);

	opp_v = neigh->opp_vertex[next_el[edge_no][0]];
	neigh = neigh->neigh[next_el[edge_no][0]];
      }
      else {
	bound = new_bound_val_3d(neigh->face_bound[next_el[edge_no][1]],bound);

	opp_v = neigh->opp_vertex[next_el[edge_no][1]];
	neigh = neigh->neigh[next_el[edge_no][1]];
      }
    }
  }

  for (j = 0; j < *n_neigh; j++)
    *(list_bound[j]) = bound;

  MEM_FREE(list_bound, max_no_list_el, LIST_BOUND);
  return(1);
}

/*--------------------------------------------------------------------------*/
/* fill_more_bound_info_3d(): This function fills boundary information using*/
/* the function new_edge_3d() above. It also counts faces and edges.        */
/*--------------------------------------------------------------------------*/

static void fill_more_bound_info_3d(MESH *mesh, const MACRO_DATA *data)
{
  /* FUNCNAME("fill_more_bound_info_3d"); */
  int        i, k, node, ne = mesh->n_elements;
  int        n_edges = 0, n_faces = 0, max_n_neigh = 0, n_neigh;
  MACRO_EL   *mel = mesh->macro_els, *neigh;

  mesh->max_edge_neigh = 8;
  node = mesh->node[FACE];

  for (i = 0; i < ne; i++) {
    for (k = 0; k < N_EDGES_3D; k++) {
/*--------------------------------------------------------------------------*/
/*  check for not counted edges                                             */
/*--------------------------------------------------------------------------*/
      n_neigh = 1;

      if (new_edge_3d(data, mel+i, k, &n_neigh)) {
	n_edges++;
	max_n_neigh = MAX(max_n_neigh, n_neigh);
      }
    }

    for (k = 0; k < N_NEIGH_3D; k++) {
      neigh = mel[i].neigh[k];
/*--------------------------------------------------------------------------*/
/*  face is counted by the element with bigger index                        */
/*--------------------------------------------------------------------------*/
      if (neigh && (neigh->index > mel[i].index))  continue;
      
      n_faces++;
    }
  }

  mesh->max_edge_neigh = MAX(mesh->max_edge_neigh, 2*max_n_neigh);
  mesh->n_edges = n_edges;
  mesh->n_faces = n_faces;

  return;
}

/*--------------------------------------------------------------------------*/
/* AI_get_orientation(): This function calculates an elements orientation   */
/* and returns it as a signed integer suitable for the MACRO_EL orientation */
/* field. Also called from read_mesh.c                                      */
/*--------------------------------------------------------------------------*/

S_CHAR AI_get_orientation(MACRO_EL *mel)
{
  int i,j;
  double a[3][3], s;

  for (i = 0; i < 3; i++)
    {
      for (j=0; j<3; j++) 
	{
	  a[i][j] = mel->coord[i+1][j] - mel->coord[0][j];
	}
    }

  s = (a[0][1]*a[1][2] - a[0][2]*a[1][1])*a[2][0]
    + (a[0][2]*a[1][0] - a[0][0]*a[1][2])*a[2][1]
    + (a[0][0]*a[1][1] - a[0][1]*a[1][0])*a[2][2];
  
  return (s >= 0) ? 1 : -1;
}

/*--------------------------------------------------------------------------*/
/* macro_test_3d(): Check data for potential cycles during refinement       */
/* At the moment, a correction (and subsequent writing of a correct macro   */
/* data file) can only be done in 2D.                                       */
/*--------------------------------------------------------------------------*/

#if 0
ALBERTA_UNUSED(static int check_cycles_3d(MACRO_DATA *));
#endif

static void macro_test_3d(MACRO_DATA *data, const char *new_name)
{
  FUNCNAME("macro_test_3d");
#if 1
  WARNING("not implemented for 3d yet: no check is performed\n");
#else
  U_CHAR error_found = false;
  int i = -1;

  i = check_cycles_3d(data);

  if (i >= 0)
  {
    error_found = true;
    WARNING("There is a cycle beginning in macro element %d.\n", i);
    ERROR_EXIT("Correction of cycles not implemented yet in 3D\n");
  }

  if (error_found && new_name) {
    MSG("Attempting to write corrected macro data to file %s...\n", new_name);
    write_macro_data(data, new_name);
  }
#endif
  return;
}
