/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_file_manager.c

   File implements "list of selected files" component and some helper functions.

   List of selected files is primarily a data structure.  It is
   displayed in a window embedded in main application window.  User
   can use Up/Down arrow keys, Home/End keys, PageUp/PageDown keys to
   move on this list, and Delete key to deselect files from the
   list. Files selected by user will be burned to optical disc or
   written to ISO image.

   List of selected files (the data structure) can be accessed by
   other parts of application, e.g. by mkisofs-related code to create
   graftpoints file.
*/

#define _BSD_SOURCE /* strdup() */

#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stddef.h>

#include "cdw_main_window.h"
#include "gettext.h"
#include "cdw_file.h"
#include "cdw_fs.h"
#include "cdw_file_manager.h"
#include "cdw_string.h"
#include "cdw_list_display.h"
#include "cdw_ncurses.h"
#include "cdw_widgets.h"
#include "cdw_window.h"
#include "cdw_debug.h"
#include "cdw_fs_browser.h"
#include "cdw_graftpoints.h"
#include "cdw_config.h"
#include "cdw_logging.h"
#include "canonicalize.h"
#include "cdw_file_selector.h"


extern cdw_config_t global_config;


typedef struct {
	long long size; /** \brief Size of currently selected files */
	double size_mb; /** \brief Size of currently selected files (in megabytes) */
	bool has_4GB_file;
	CDW_LIST_DISPLAY *display;

	bool follow_symlinks;
} cdw_selected_files_t;


static cdw_selected_files_t selected_files;
static cdw_file_selector_t *file_selector = (cdw_file_selector_t *) NULL;


static void     cdw_file_manager_internal_error_dialog(int error);

static cdw_rv_t cdw_file_selector(void);
static cdw_rv_t cdw_file_manager_add_file_to_selected(cdw_file_selector_t *selector, cdw_selected_files_t *selected);
static bool     cdw_file_manager_file_properties_is_appendable(cdw_file_t const * file);

static void     cdw_selected_files_init(void);
static cdw_rv_t cdw_selected_files_append_copy(cdw_selected_files_t *selected, cdw_file_t const * file);
static cdw_rv_t cdw_selected_files_remove_file(cdw_selected_files_t *selected, size_t i);


enum {
	CDW_FM_E_UNEXPECTED,     /* unexpected error, generic error message */
	CDW_FM_E_NOT_ADDED,      /* file was not added to list of selected files */
	CDW_FM_E_NO_SELECTOR,    /* can't show file selector for some reason */
	CDW_FM_E_NO_SELECTED     /* can't show list of selected files */
};





/**
   \brief Initialization function for "selected files" part of "file manager"

   To be called during start of application by cdw_file_manager_init().
*/
void cdw_selected_files_init(void)
{
	selected_files.size_mb = 0.0;
	selected_files.size = 0;
	selected_files.has_4GB_file = false;
	selected_files.display = (CDW_LIST_DISPLAY *) NULL;

	/* The function is called during initialization of cdw, so
	   this global variable holds the most recent value. */
	selected_files.follow_symlinks = global_config.general.selected_follow_symlinks;

	return;
}





/**
   \brief Initialize file manager module

   \date Function's top-level comment reviewed on 2016-02-19
   \date Function's body reviewed on 2016-02-19

   Function initializes "selected files" and "file selector"
   components used by file manager.

   You should initialize fs module (by calling cdw_fs_init()) before
   calling this function.

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_manager_init(void)
{
	cdw_selected_files_init();

	file_selector = cdw_file_selector_new();
	if (!file_selector) {
		cdw_vdm ("ERROR: failed to create file selector in file manager module\n");
		return CDW_ERROR;
	}
	return CDW_OK;
}





/**
   \brief Clean up resources used by file manager module

   \date Function's top-level comment reviewed on 2016-02-19
   \date Function's body reviewed on 2016-02-19

   Call this function to deallocate all resources used by file manager
   module when closing the program.
*/
void cdw_file_manager_clean(void)
{
	cdw_selected_files_delete_view();

	if (file_selector) {
		cdw_file_selector_hide(file_selector);
		cdw_file_selector_delete(&file_selector);
	}

	return;
}





/**
   \brief Return pointer to list of selected files

   Function returns pointer to head of doubly linked list
   of selected files (files selected by user to be burned to
   optical disc or written to ISO file).

   'selected_files' object should be initialized before calling
   this function.

   \return pointer to list of selected files, may be NULL if the list is empty
*/
cdw_dll_item_t *cdw_selected_files_get_list(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"selected files\" display is null\n");
	cdw_assert (selected_files.display->list != (cdw_dll_item_t *) NULL, "ERROR: list of selected files is null\n");

	return selected_files.display->list;
}





/**
   \brief Set new setting regarding "follow selected symbolic links" policy

   \param follow
*/
void cdw_selected_files_set_follow_symlinks(bool follow)
{
	selected_files.follow_symlinks = follow;

	return;
}





/**
   \brief Top level function for process of selecting files from file system

   Function displays file selector widget, in which user can select
   files/directories from file system, adds the files to list of
   selected files, and refreshes information (displayed in main app window)
   about currently selected files.

   'selected_files' display should be already initialized with
   cdw_selected_files_create_view().

   'file_selector' also needs to be initialized before calling this function.

   \return CDW_OK if some files were selected without problems
   \return CDW_NO if no files were selected (and no problems occurred)
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_manager_handle_adding_to_selected_files(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display not initialized properly\n");
	cdw_assert (file_selector->initial_fullpath != (char *)  NULL, "ERROR: initial dirpath is null\n");

	/* Creating selector window, calling driver, adding files to
	   list, etc. is done inside of this function. */
	cdw_rv_t crv = cdw_file_selector();
	cdw_main_window_wrefresh();

	cdw_rv_t retval = CDW_ERROR;
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: file selector returns CDW_ERROR\n");
		retval = CDW_ERROR;
	} else {
		if (selected_files.display->n_items == 0) {
			cdw_sdm ("INFO: selected_files->n_items == 0\n");
			retval = CDW_NO;
		} else {
			retval = CDW_OK;
		}
	}
	/* scroll()/refresh() even in case of errors: error may have occurred
	   when selecting 2nd or 3rd file, but the files that were selected
	   correctly must be displayed - using scroll() */
	cdw_list_display_scroll_to(selected_files.display, 0, 0, false);
	cdw_list_display_refresh(selected_files.display);

	return retval;
}





/**
   \brief Top level function for process of deselecting files from list of selected files

   Function moves keyboard focus to view with list of selected files
   and allows user to use movement keys to move on the list, and to
   use Delete key to deselect items from list of selected files.

   Information in 'selected files info' view in main app window is
   updated every time a file is deselected.

   Function displays error message if some problem occurred.

   \return CDW_OK on success
   \return CDW_ERROR on problems with deselecting file.
*/
cdw_rv_t cdw_file_manager_handle_deleting_from_selected_files(void)
{
	int key = 'a';
	cdw_rv_t crv = CDW_OK;

	while (selected_files.display->n_items > 0) {
		key = cdw_list_display_driver(selected_files.display);
		cdw_assert (selected_files.display->n_items > selected_files.display->current_item_ind,
			    "ERROR: file index is larger than number of files\n");
		if (key == KEY_DC) {
			crv = cdw_selected_files_remove_file(&selected_files, selected_files.display->current_item_ind);
			if (crv == CDW_OK) {
				crv = cdw_main_window_volume_info_view_update(-1, -1, false);
				if (crv == CDW_OK) {
					crv = CDW_OK;
				} else {
					cdw_vdm ("ERROR: failed to update files info view with fetch_data=false\n");
					crv = CDW_ERROR;
					break;
				}
			} else {
				cdw_vdm ("ERROR: failed to remove file nr %zd / %zd from 'selected files'\n",
					 selected_files.display->current_item_ind, selected_files.display->n_items);
				cdw_file_manager_internal_error_dialog(CDW_FM_E_UNEXPECTED);
				crv = CDW_ERROR;
				break;
			}
		} else if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
			/* 'Q' = quit. Exit "selected files" view. */
			break;
		} else {
			;
		}
	}
	/* FIXME: this function does not inform user when some errors occur */

	crv = cdw_main_window_volume_info_view_update(-1, -1, true);
	if (crv == CDW_OK) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to update files info view with fetch_data=false\n");
		return CDW_ERROR;
	}
}





/**
   \brief Get number of files that are currently selected by user

   Function returns number of files in 'selected files' component.
   'selected files' component need to be initialized before calling
   this function.

   \return number of files on 'selected files' list
*/
size_t cdw_selected_files_get_number(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "'selected files' object not initialized\n");

	return selected_files.display->n_items;
}





/**
   \brief Display a window with file system browser

   \date Function's top-level comment reviewed on 2016-02-19
   \date Function's body reviewed on 2016-02-19

   Function displays new window with list of files in initial directory.
   User can use movement keys to move on the list, and Enter key to change
   directory - like in any file manager. User can't use Delete key to remove
   items from the list, but can use Space key to select them. Every selected
   file is added to 'selected files' list, so that later it can be burned
   to disc or written to ISO file.

   Function updates 'selected files info' area whenever new file is selected.
   It also updates number of selected files accordingly.

   User can close the window by pressing Escape or Q or q key.

   The function modifies selector->initial_fullpath, so that file
   selector remembers last directory which was visited during last
   call of the function (last time that the user selected files).

   In case of errors function returns with CDW_ERROR, but does not erase
   list of already selected files.

   \return CDW_OK when no errors occurred during selection of files
   \return CDW_ERROR on errors in process of selecting files
*/
cdw_rv_t cdw_file_selector(void)
{
	cdw_assert (file_selector, "ERROR: file selector is not initialized\n");

	/* Size and location of file selector window is selected so that:
	   - there is enough space at the bottom so "volume info" view
             is visible (but not it's upper border, hence +1);
	   - the window is big enough to cover whole list of selected
	     files displayed in main app window, so we don't have to
	     refresh it every time a file is selected. */
	cdw_file_selector_show(file_selector, NULL,
			       LINES - (CDW_UI_VOLUME_VIEW_N_ROWS + CDW_UI_TOOLTIPS_VIEW_N_ROWS) + 1,
			       COLS,
			       0, 0,  /* begin_y, begin_x */
			       CDW_COLORS_DIALOG,
			       /* 2TRANS: this is title of file selector window */
			       _("Select files/directories"),
			       NULL); /* fullpath */
	cdw_widget_add_return_keys_pointer(&file_selector->widget, CDW_KEY_SPACE, 0);

	cdw_rv_t retval = CDW_OK;
	int key = 'a';
	do {
		key = cdw_file_selector_driver(file_selector);
		if (key == -1) {
			retval = CDW_ERROR;
		} else if (key == CDW_KEY_SPACE) {
			cdw_file_manager_add_file_to_selected(file_selector, &selected_files);
		} else {
			;
		}
	} while (key == CDW_KEY_SPACE);
	cdw_file_selector_hide(file_selector);

	if (selected_files.display->n_items > 0) {
		/* true - upon finishing adding files recount size of
		   selected files */
		if (CDW_OK != cdw_main_window_volume_info_view_update(-1, -1, true)) {
			retval = CDW_ERROR;
			cdw_vdm ("ERROR: failed to update files info view with fetch_data=true\n");
		}
	}

	return retval;
}





/**
   \brief Create widget displayed in 'selected files' area of main app window

   Function creates widget with list of files (initially empty) and puts
   it in given \p parent window. \p parent should be initialized ncurses
   window, part of main application window. Widget created by this function
   will be 'embedded' in this window.

   \param parent - window into which new widget will be put

   \return CDW_OK on success
   \return CDW_ERROR when widget was not created properly
*/
cdw_rv_t cdw_selected_files_create_view(WINDOW *parent)
{
	/* these are parameters for embedded selector window */
	int n_lines = getmaxy(parent);
	cdw_assert (n_lines > 2, "ERROR: parent has no more than 2 lines\n");
	int n_cols = getmaxx(parent);
	cdw_assert (n_cols > 2, "ERROR: parent has no more than 2 columns\n");

	cdw_assert (selected_files.display == (CDW_LIST_DISPLAY *) NULL,
		    "'selected_files' display is already initialized\n");

	/* 1, 1 - begin_y, begin_x */
	selected_files.display = cdw_list_display_new(parent, n_lines - 2, n_cols - 2, 1, 1, CDW_COLORS_MAIN);
	if (!selected_files.display) {
		cdw_vdm ("ERROR: failed to create 'selected files' display\n");
		cdw_file_manager_internal_error_dialog(CDW_FM_E_NO_SELECTED);

		return CDW_ERROR;
	} else {
		selected_files.display->display_item = cdw_file_display_file;
		cdw_list_display_add_return_key(selected_files.display, KEY_DC);
		selected_files.display->display_format = CDW_LIST_DISPLAY_FORMAT_LONG;

		cdw_list_display_refresh(selected_files.display);

		return CDW_OK;
	}
}





void cdw_selected_files_delete_view(void)
{
	if (selected_files.display) {
		cdw_file_dealloc_files_from_list(selected_files.display->list);
		cdw_list_display_delete(&(selected_files.display));
	}

	return;
}





/**
   \brief Display dialog window with error message

   Function displays dialog window with one of several error messages.
   Error messages are chosen based on value of \p error. All dialog windows
   have only "OK" button.

   \param error - id of an error, accepted values are defined on top of this file
*/
void cdw_file_manager_internal_error_dialog(int error)
{
	/* gettext(): "resulting string [...] must not be modified or freed" */
	char *message = (char *) NULL;
	if (error == CDW_FM_E_NOT_ADDED) {
		/* 2TRANS: this is message in dialog window */
		message = _("File not added. Please try again.");
	} else if (error == CDW_FM_E_UNEXPECTED) {
		/* 2TRANS: this is message in dialog window */
		message = _("Unexpected error. Please close this window and open it again.");
	} else if (error == CDW_FM_E_NO_SELECTOR) {
		/* 2TRANS: this is message in dialog window. Some
		   function returned error value. No further action
		   will be performed, no other explanations provided */
		message = _("Cannot show file selector window. Some error occurred.");
	} else if (error == CDW_FM_E_NO_SELECTED) {
		/* 2TRANS: this is message in dialog window. Some
		   function returned error value. No further action
		   will be performed, no other explanations provided */
		message = _("Cannot display view with selected files. You should restart application.");
	} else {
		return;
	}

	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("File manager error"),
			   message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);

	return;
}





/**
   \brief Refresh 'selected files' widget that is displayed in main application window

   Function redraws list of selected files, the list displays files from
   first to whichever still fits into widget's window.

   \return CDW_OK
*/
cdw_rv_t cdw_selected_files_regenerate_view(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL,
		    "you called the function too early, selected_files is not initialized yet\n");
	cdw_list_display_scroll_to(selected_files.display, 0, 0, false);

	return CDW_OK;
}





/**
   \brief Count size (in file system) of all items on 'selected files' list

   \date Function's top-level comment reviewed on 2016-02-19
   \date Function's body reviewed on 2016-02-19

   On success the size of selected files in megabytes is returned.

   The function is called "_calculate_" to indicate that the size is
   counted every time the function is called. If you are sure that
   size of selected files was not changed since last time you
   calculated it, you can call long long cdw_selected_files_get_size()
   instead.

   TODO: if one of selected files is removed in native file system, and then
   this function is called, it detects that a file is missing and returns -1.
   The function should somehow mark files on selected files list that are
   missing, and mark them somehow.

   \return size (in megabytes) of all files on 'selected list' on success
   \return -1 on error
*/
double cdw_selected_files_calculate_size_mb(void)
{
	cdw_fs_visitor_data_t vdata;
	vdata.size = 0;
	vdata.has_file_over_4GB = false;
	vdata.follow_symlinks = selected_files.follow_symlinks;
	selected_files.size_mb = 0.0;
	selected_files.size = 0;

	size_t i = 0;
	for (cdw_dll_item_t *f = selected_files.display->list; f; f = f->next) {
		cdw_file_t *file = (cdw_file_t *) f->data;
		cdw_assert(file, "ERROR: file #%zd is NULL\n", i);
		cdw_assert(file->fullpath != (char *) NULL, "ERROR: file #%zd has no fullpath\n", i);

#ifndef NDEBUG
		fprintf(stderr, "\n\n");
#endif
		cdw_vdm ("INFO: traversing file \"%s\"\n", file->fullpath);

		cdw_rv_t crv = cdw_fs_traverse_path(file->fullpath, cdw_fs_visitor, &vdata);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to get dirsize of \"%s\" item\n", file->fullpath);
			return -1; /* error value */
		} else {
			/* this is to update size displayed next to file name
			   in "selected files" view - the size may have changed
			   after toggling "selected_follow_symlinks" flag */
			file->size = vdata.size;

			selected_files.size += vdata.size;
			vdata.size = 0;
		}
		i++;
	}

	selected_files.size_mb = (((double) selected_files.size) / 1024.0) / 1024.0;
	return selected_files.size_mb;
}





/**
   \brief Append copy of given file at the end of list of selected files

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   The function does not refresh displayed information, you have to call
   cdw_list_display_scroll() to display new, updated content.

   The function will not append a file if it is invalid, unsupported, or if
   it is reference to parent dir ("..").

   The function will not append a file if it already exist on a list of selected files.

   The function creates a copy of \p file, and the copy is appended to
   display.

   This function was created to wrap in one function the code responsible
   for adding file to list of selected files. Therefore it is quite
   specialized (e.g. can't append "..").

   \param selected - object with list of already selected files
   \param file - file that you want to append to a display with list of selected files

   \return CDW_NO if no action was taken, but no error occurred either
   \return CDW_ERROR on errors
   \return CDW_OK if file was appended correctly
*/
cdw_rv_t cdw_selected_files_append_copy(cdw_selected_files_t *selected, cdw_file_t const * file)
{
	cdw_assert (selected->display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"selected files\" display is NULL\n");
	cdw_assert (file, "ERROR: \"file\" argument is NULL\n");
	cdw_assert (file->fullpath != (char *) NULL, "ERROR: file has no fullpath set\n");

	cdw_assert (cdw_file_manager_file_properties_is_appendable(file),
		    "ERROR: file is not appendable, you should have caught it before calling this function\n");

	if (!cdw_file_manager_file_properties_is_appendable(file)) {
		/* TODO: This is already done in
		   cdw_file_manager_add_file_to_selected() and caught
		   in assert() above. You can remove this condition in
		   future. */
		return CDW_NO;
	}

	cdw_file_t *copy = cdw_file_duplicate(file);
	if (!copy) {
		cdw_vdm ("ERROR: failed to duplicate file %ld / \"%s\"\n", (long int) file, file->fullpath);
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_list_display_add_item(selected->display, (void *) copy, cdw_file_equal);

	if (crv == CDW_OK) {
		selected->size += copy->size; /* this is to avoid counting size of all selected files again */
		cdw_vdm ("INFO: added file with size = %zd, fullpath = \"%s\"\n", copy->size, copy->fullpath);

	} else if (crv == CDW_NO) {
		/* given file was already on a list, no need to do anything */
		cdw_vdm ("INFO: not added file with path \"%s\", was already on list\n", copy->fullpath);
	} else {
		cdw_vdm ("ERROR: failed to add file \"%s\"\n", copy->fullpath);
		cdw_file_manager_internal_error_dialog(CDW_FM_E_NOT_ADDED);
		/* even if adding fails, nothing really happens, this
		   warning was all we could do */
	}

	return crv;
}





/**
   \brief Remove i-th item from "selected files" object

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   Function looks up \p i-th item in "selected files" object, removes it
   from associated window and associated list. It also updates value of "number
   of files" variable associated with the object.

   \param selected - object with list of already selected files
   \param i - index of file to be removed

   \return CDW_OK if removing was successful
   \return CDW_ERROR if it failed
*/
cdw_rv_t cdw_selected_files_remove_file(cdw_selected_files_t *selected, size_t i)
{
	cdw_assert (selected->display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display is NULL\n");
	cdw_assert (selected->display->n_items > 0, "ERROR: you called the function for empty list of selected files\n");
	cdw_assert (selected->display->n_items > i, "ERROR: index of file is larger than number of files in \"selected files\" display\n");
	cdw_assert (selected->display->list != (cdw_dll_item_t *) NULL, "ERROR: list associated with the display is NULL\n");

	cdw_dll_item_t *item = cdw_dll_ith_item(selected->display->list, i);
	if (!item) {
		cdw_vdm ("ERROR: item #%zd not found on list of selected files\n", i);
		return CDW_ERROR;
	}

	cdw_file_t *file = (cdw_file_t *) item->data;
	selected->size -= file->size;
	if (selected->size < 0) {
		cdw_vdm ("ERROR: size of selected files dropped to negative value %lld\n", selected->size);
		selected->size = 0;
	}
	/* Notice that here we are freeing item->data, not file. */
	cdw_file_delete((cdw_file_t **) &(item->data));

	cdw_rv_t crv = cdw_list_display_remove_item(selected->display, i);

	if (crv == CDW_OK) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to remove item #%zd from list of selected files\n", i);
		return CDW_ERROR;
	}
}





cdw_rv_t cdw_file_manager_create_graftpoints_file(void)
{
	if (selected_files.display->n_items == 0){
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("No files selected"),
				   /* 2TRANS: this is message in dialog window, user
				      wants to write files or create image, but no
				      files from hdd are selected yet */
				   _("No files selected. Please use 'Add files'"),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_NO;
	} else {
		cdw_rv_t crv = cdw_graftpoints_create_file();
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create graftpoints file\n");
			return CDW_ERROR;
		} else {
			return CDW_OK;
		}
	}

}




void cdw_file_manager_delete_graftpoints_file(void)
{
	cdw_graftpoints_delete_file();

	return;
}





bool cdw_selected_files_file_over_4gb_present(void)
{
	return selected_files.has_4GB_file;
}




/* TODO: change return type to off_t. */
long long cdw_selected_files_get_size(void)
{
	return selected_files.size;
}





/**
   \brief Add currently highlighted file to list of selected files

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   Add a file currently selected in \p selector to list of selected files \p selected.

   \param selector - file selector
   \param selected - object with selected files

   \return CDW_OK on success
   \return CDW_NO if file cannot be appended (but no error occurred)
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_manager_add_file_to_selected(cdw_file_selector_t *selector, cdw_selected_files_t *selected)
{
	cdw_assert (selector->fs_browser->display->n_items > selector->fs_browser->display->current_item_ind,
		    "ERROR: index of file is larger than number of files: n_items = %zd, item_i = %zd\n",
		    selector->fs_browser->display->n_items, selector->fs_browser->display->current_item_ind);


	cdw_fs_browser_debug_report_current_fullpath(selector->fs_browser, "file manager: adding file");


	cdw_file_t * const file = cdw_fs_browser_get_current_file(selector->fs_browser);
	cdw_assert (file, "ERROR: failed to get current file from fs browser\n");


	if (!cdw_file_manager_file_properties_is_appendable(file)) {
		return CDW_NO;
	} else if (cdw_list_display_is_member(selected->display, (void *) file, cdw_file_equal)) {
		/* file is already on list of selected files */
		cdw_vdm ("INFO: file \"%s\" already selected, not adding\n", file->fullpath);
		return CDW_NO;
	} else {
		; /* Add the file. */
	}


	cdw_fs_visitor_data_t vdata;
	vdata.size = 0;
	vdata.has_file_over_4GB = false;
	vdata.follow_symlinks = selected->follow_symlinks;
	cdw_rv_t crv = cdw_fs_traverse_path(file->fullpath, cdw_fs_visitor, &vdata);
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("For some reason cdw can't visit all files in selected directory (consult log file for details)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		/* 2TRANS: this is message printed into log file;
		   %s is a full path to a file */
		cdw_logging_write(_("ERROR: failed to fully scan this path: \"%s\"\n"), file->fullpath);

		return CDW_ERROR;
	}


	file->size = vdata.size;
	crv = cdw_selected_files_append_copy(selected, file);
	if (crv == CDW_NO) {
		/* item not appendable (i.e. not file dir, or link) */
		return CDW_NO;
	} else if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: failed to append file to selected files\n");
		return CDW_ERROR;
	} else if (crv != CDW_OK) {
		cdw_assert (0, "unhandled return value %d\n", crv);
	} else {
		; /* File appended. */
	}


	/* false - don't recount size of selected files (yet), size of
	   selected files was updated in
	   cdw_selected_files_append_copy(), and will be recounted
	   again on file selector exit. */
	crv = cdw_main_window_volume_info_view_update(-1, -1, false);
	/* Updating files info view may disrupt part of file selector
	   window; redraw file selector. */
	cdw_list_display_refresh(selector->fs_browser->display);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to update files info view with fetch_data=false\n");
		return CDW_ERROR;
	}


	/* crv == CDW_OK, file was added to list of selected files.
	   Save current location so that next time user opens file
	   selector, then working directory and highlighted file is
	   the same as when user previously used the selector. fs
	   browser can handle situation in which previously visited
	   dir / file disappears between two calls to fs browser. */
	char *fullpath = cdw_fs_browser_get_current_fullpath(selector->fs_browser);
	cdw_assert (fullpath, "ERROR: failed to get current dirpath\n");
	cdw_string_set(&selector->initial_fullpath, fullpath);
	CDW_STRING_DELETE (fullpath);

	cdw_vdm ("INFO: saved last file fullpath as %s\n", selector->initial_fullpath);
	selector->first_selection = false;


	return CDW_OK;
}





/**
   \brief Decide if given file can be appended to list of selected files based on file's properties

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   The function only check properties of the file itself. The file may
   be OK, but caller still may want to not to append the file because
   it already exists on list of selected files. It is up to caller to
   do any additional checks.

   \param file - file to be checked

   \return true if file can be appended to list of selected files
   \return false otherwise
*/
bool cdw_file_manager_file_properties_is_appendable(cdw_file_t const * file)
{
	if (file->invalid) {
		cdw_vdm ("INFO: invalid file is not appendable\n");
		return false;

	} else if (file->is_ref_to_parent_dir) {
		/* Never allow selecting parent dir. */
		cdw_vdm ("INFO: parent dir is not appendable\n");
		return false;

	} else if (!(file->type & CDW_FS_FILE) && !(file->type & CDW_FS_DIR)) {
		/* Only directories and regular files (or links to them - notice the '&' operator) allowed. */
		cdw_vdm ("INFO: file of strange type \"0x%x\" is not appendable\n", file->type);
		return false;

	} else {
		return true;
	}
}





/**
   \brief Look for \p path on list of selected files

   \date Function's top-level comment reviewed on 2016-02-20
   \date Function's body reviewed on 2016-02-20

   This is just a very simple implementation. It does not attempt to
   follow symbolic links that may be in selected files list.

   The \p path is canonicalized before function starts looking for it
   on list of selected files.

   \p can_mode is canonicalization mode, as seen in
   gnulib/lib/canonicalize.h. Not completely sure if this argument is
   useful at all.

   TODO: improve the function so that it covers more cases.

   \param path - path to look for on list of selected files
   \param can_mode - canonicalization mode

   \return CDW_OK if path is on list of selected files
   \return CDW_NO if path is not on list of selected files
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_selected_files_search(char const * path, int can_mode)
{
	char *fullpath = canonicalize_filename_mode(path, can_mode);
	if (!fullpath) {
		cdw_vdm ("ERROR: failed to canonicalize path \"%s\"\n", path);
		return CDW_ERROR;
	} else {
		cdw_vdm ("INFO: canonicalized: \"%s\"\n", fullpath);
	}

	/* Check if a file with given fullpath exists on list of selected files. */
	bool is_member = cdw_list_display_is_member(selected_files.display, (void const *) fullpath, cdw_file_equal_by_fullpath);

	CDW_STRING_DELETE (fullpath);

	return is_member ? CDW_OK : CDW_NO;
}
