/*
 * Copyright (C) 2009, 2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <diagnostics/extensions/stacktrace/descriptor.hpp>
#include <diagnostics/extensions/stacktrace/error.hpp>

#include <unistd.h>
#include <fcntl.h>

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

void
POSIX::Descriptor::OpenReadOnly(const char *path)
{
  Descriptor tmp(::open(path, O_RDONLY, 0));
  Error::Unless(tmp.IsOpen());
  Close(); // explicit close to propagate exception.
  swap(tmp);
}

//void
//POSIX::Descriptor::OpenReadOnly(const Descriptor &dir, const char *path)
//{
//  Descriptor tmp(::openat(dir.GetRaw(), path, O_RDONLY, 0));
//  Error::Unless(tmp.IsOpen());
//  Close(); // explicit close to propagate exception.
//  swap(tmp);
//}

void
POSIX::Descriptor::Close()
{
  if (IsOpen()) {
    const int d = desc;
    desc = -1;
    Error::Unless(::close(d) == 0);
  }
}

void
POSIX::Descriptor::CloseNoThrow() throw ()
{
  if (IsOpen()) {
    ::close(desc);
    desc = -1;
  }
}

size_t
POSIX::Descriptor::read(void *buffer, size_t len)
{
  ssize_t ret = ::read(desc, buffer, len);
  Error::Unless(ret >= 0);
  return ret;
}

void
POSIX::Descriptor::SetCloseOnExec(bool on)
{
  int ret = fcntl(GetRaw(), F_GETFD, 0);
  Error::Unless(ret >= 0);
  long flag;
  if (on) {
    flag = ret | FD_CLOEXEC;
  } else {
    flag = ret & ~FD_CLOEXEC;
  }
  Error::Unless(fcntl(GetRaw(), F_SETFD, flag) >= 0);
}

void
POSIX::Descriptor::SetNonBlocking(bool on)
{
  int ret = fcntl(GetRaw(), F_GETFL, 0);
  Error::Unless(ret >= 0);
  long flag;
  if (on) {
    flag = ret | O_NONBLOCK;
  } else {
    flag = ret & ~O_NONBLOCK;
  }
  Error::Unless(fcntl(GetRaw(), F_SETFL, flag) >= 0);
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

