// license:BSD-3-Clause
// copyright-holders: Andreas Naive, David Haywood

#include "emu.h"
#include "igs036crypt.h"

/****************************************************************************
IGS036 encryption emulation

The encryption used by the IGS036 seems to be another iteration over
previous IGS encryption schemes. Basically, it consists on a rotation-based
non-trivial obfuscation layered upon a simple address-based XOR encryption
(similar to the ones found in previous IGS circuits).

The scheme works on 16-bits words and is probably designed to depend on 24 bits of
(word-) address; in what follows, we will refer to the 8 lowest ones simply as the
lowest bits of the address, and the other 16 as the highest bits the address.

The address-based XOR can be thought as 16 one-bit XORs against key bits
controlled by certain combinations of up to three address bits. The game key is
comprised of 256 8-bits values provided by the internal ARM; every 8-bit value
in the key is used on those words whose address match the index modulus 256;
in a given key byte, every bit affects two positions of the corresponding 16-bits
encrypted words.
This use of the key is similar to the one found in previous instantiations of
IGS circuits.

What is new in the IGS036 is the use of an obfuscation working this way:

1) The highest bits of the address are split in 4 groups, every of which controls
a rotation by a shift of (plus or minus) 9, 1, 2 and 4 respectively.
2) For every address, the highest bit of the group set in the address controls
the activation/deactivation of the rotation for his group, using an associated
(and fixed) boolean function depending on the lowest bits of the address.
3) If the group rotation is to be activated according to 2), then another
fixed group-level boolean functions (again, depending on the lowest bits of the
address) control the direction (left or right) of the rotation.
4) One of the groups (the associated with the shift by 9) interacts with the other
three by inverting (when active itself) the activation/deactivation patterns of
the other three.
5) The lowest bits of the address control a further rotation(independent
on the highest bits of the address).
6) Finally, a global bitswap is applied.

All the associated boolean functions are clearly of low complexity, so it should
be expected that the hardware is calculating them that way rather than using
lookup tables or otherwise. It should be stressed that this obfuscation is done
system-wide without dependence on the game keys.

On a different note, the unused tail of the ROMs are pattern-filled and, more interestingly,
that region appears to be hiding 20-bytes values (SHA-1 hashes?) located at
positions which vary per set. See the table below.

driver      20-bytes value position in the ROM
---------   ----------------------------------
orleg2o     $763984-$763997
orleg2      $76C77C-$76C78F
kov3        $718040-$718053
kof98umh    $E50E60-$E50E73

TO-DO:
- complete the table with the 20-bytes values
- decryption tables for the programs using less than 0x80000 of code (huoqilin, jhzb)

*****************************************************************************/

igs036_decryptor::igs036_decryptor(const uint8_t *game_key)
	: key(game_key)
{
}

void igs036_decryptor::decrypter_rom(uint16_t *rom, int size, int offset)
{
	for (int i = 0; i < size / 2; i++)
	{
		rom[i] = decrypt(rom[i], i + offset);
	}
}

uint16_t igs036_decryptor::decrypt(uint16_t cipherword, int word_address) const
{
	// key-independent manipulation
	int aux = deobfuscate(cipherword, word_address);

	// key-dependent manipulation
	for (int i = 0; i < 16; ++i)
	{
		if ((word_address & triggers[i][0]) == triggers[i][1])
			aux ^= BIT(key[word_address&0xff], i & 7) << i;
		else
			aux ^= BIT(0x1a3a, i) << i;
	}

	return aux;
}

uint16_t igs036_decryptor::deobfuscate(uint16_t cipherword, int word_address) const
{
	// key-independent manipulation
	const int shift = rotation(word_address);
	int aux = rol(cipherword, shift);
	aux = bitswap<16>(aux, 10,9,8,7,0,15,6,5,   14,13,4,3,12,11,2,1);

	return aux;
}

int igs036_decryptor::rotation(int address) const
{
	const int group15[] = {15,11, 7, 5};  // 15 is a guess
	const int group14[] = {14, 9, 3, 2};
	const int group13[] = {13,10, 6, 1};
	const int group12[] = {12, 8, 4, 0};

	// rotation depending on all the address bits
	const int enabled0 = rot_enabled(address, group15);
	int rot = enabled0 * rot_group(address, group15) * 9;

	const int enabled1 = enabled0 ^ rot_enabled(address, group14);
	rot += enabled1 * rot_group(address, group14) * 1;

	const int enabled2 = enabled0 ^ rot_enabled(address, group13);
	rot += enabled2 * rot_group(address, group13) * 2;

	const int enabled3 = enabled0 ^ rot_enabled(address, group12);
	rot += enabled3 * rot_group(address, group12) * 4;

	// block-independent rotation (just depending on the lowest 8 bits)
	int rot2  = 4 * BIT(address, 0);
	rot2 += 1 * BIT(address, 4) * (BIT(address, 0) * 2 - 1);
	rot2 += 4 * BIT(address, 3) * (BIT(address, 0) * 2 - 1);
	rot2 *= (BIT(address, 7) | (BIT(address, 0) ^ BIT(address, 1) ^ 1)) * 2 - 1;
	rot2 += 2 * ((BIT(address, 0) ^ BIT(address, 1)) & (BIT(address, 7) ^ 1));

	return (rot + rot2) & 0xf;
}

int igs036_decryptor::rot_enabled(int address, const int *group) const
{
	int enabled = 0;
	for (int j = 0; j <4 ; ++j)
	{
		if (BIT(address, 8 + group[j]))
		{
			const int aux = address ^ (0x1b * BIT(address, 2));
			enabled = rot_enabling[group[j]][aux & 3](aux);
			break;
		}
	}

	return enabled;
}

int igs036_decryptor::rot_group(int address, const int *group) const
{
	const int aux = rot_direction[group[0] & 3][address & 7](address);
	return (aux * 2) - 1;
}

uint16_t igs036_decryptor::rol(uint16_t num, int shift) const
{
	const uint16_t r = num << shift;
	const uint16_t l = num >> (16 - shift);

	return r | l;
}

// the triggers describe under what conditions are every one of the 16 XORs activated

const uint32_t igs036_decryptor::triggers[16][2] = {
	{0x000101, 0x000001}, {0x000802, 0x000800}, {0x000204, 0x000004}, {0x000408, 0x000408},
	{0x010010, 0x000010}, {0x020020, 0x000020}, {0x040040, 0x000040}, {0x080080, 0x080080},
	{0x100100, 0x000100}, {0x200200, 0x200000}, {0x400400, 0x400000}, {0x800801, 0x000001},
	{0x001004, 0x001000}, {0x002010, 0x002000}, {0x004040, 0x000040}, {0x008100, 0x008100}
};


// The rotation depending on the 16 highest address bits depends on a series
// of function on the 8 lowest word-address bits. Some comments:
// * Bits #5 & #6 are unused so, in fact, they only depend on 6 address bits
// * The functions are clearly low-complexity boolean functions on those 6 bits
//   rather than, say, random lookup tables
// * There are quite a number of functionally equivalent ways to implement
//   those boolean functions, so the given implementation (by multiplexing
//   over some simple functions) shouldn't be taken too seriously: while it's
//   functionally correct, it doesn't necessarily represent the way the hardware
//   is calculating them.

static int unknown(int address) { return 0; }
static int cZero  (int address) { return 0; }
static int cOne   (int address) { return 1; }
static int bit_3  (int address) { return BIT(address, 3); }
static int bit_4  (int address) { return BIT(address, 4); }
static int bit_7  (int address) { return BIT(address, 7); }
static int not_3  (int address) { return BIT(address, 3) ^ 1; }
static int not_4  (int address) { return BIT(address, 4) ^ 1; }
static int not_7  (int address) { return BIT(address, 7) ^ 1; }
static int xor_37 (int address) { return BIT(address, 3) ^ BIT(address, 7); }
static int xnor_37(int address) { return BIT(address, 3) ^ BIT(address, 7) ^ 1; }
static int xor_47 (int address) { return BIT(address, 4) ^ BIT(address, 7); }
static int xnor_47(int address) { return BIT(address, 4) ^ BIT(address, 7) ^ 1; }
static int nor_34 (int address) { return (BIT(address, 3) | BIT(address, 4)) ^ 1; }
static int impl_43(int address) { return BIT(address, 3) || (BIT(address, 4) ^ 1); }


int (*igs036_decryptor::rot_enabling[16][4])(int)  = {
	{bit_3  , not_3  , bit_3  , not_3  },
	{bit_3  , not_3  , bit_3  , not_3  },
	{bit_4  , bit_4  , bit_4  , bit_4  },
	{bit_4  , not_4  , bit_4  , not_4  },
	{bit_3  , bit_3  , bit_3  , bit_3  },
	{nor_34 , bit_7  , bit_7  , cZero  },
	{cZero  , cOne   , cZero  , cOne   },
	{impl_43, xor_37 , xnor_37, not_3  },
	{bit_3  , bit_3  , not_3  , not_3  },
	{bit_4  , bit_4  , not_4  , not_4  },
	{cZero  , cZero  , cZero  , cZero  },
	{nor_34 , bit_7  , not_7  , cOne   },
	{bit_3  , not_3  , bit_3  , not_3  },
	{cZero  , cOne   , cOne   , cZero  },
	{bit_4  , not_4  , bit_4  , not_4  },
	{unknown, unknown, unknown, unknown},
};

int (*igs036_decryptor::rot_direction[4][8])(int)  = {
	{bit_3  , xor_37 , xnor_37, not_3  , bit_3  , xor_37 , xnor_37, not_3  },
	{cZero  , not_7  , not_7  , cZero  , cZero  , not_7  , not_7  , cZero  },
	{bit_4  , xor_47 , xnor_47, not_4  , bit_4  , xor_47 , xnor_47, not_4  },
	{bit_3  , not_7  , bit_7  , cZero  , cOne   , not_7  , bit_7  , cZero  },
};

// ------------------------GAME KEYS---------------------------

// The keys below have been obtained by an automatic process
// exploiting the simple XOR scheme used by the system. Overall, the process,
// while simple, seems to be pretty robust, so few errors should be expected,
// if any.

const uint8_t m312cn_key[0x100] = {
	0x01, 0x09, 0x02, 0xab, 0x23, 0x20, 0xa2, 0x03, 0x10, 0x9b, 0xba, 0x33, 0x04, 0x2e, 0x27, 0x23,
	0x92, 0x11, 0x13, 0x93, 0x13, 0x86, 0x83, 0x02, 0x18, 0x8a, 0x8b, 0x9a, 0x10, 0x0f, 0x13, 0x83,
	0xa2, 0x98, 0x32, 0xba, 0x06, 0xab, 0x02, 0x0b, 0x1a, 0xa0, 0x13, 0x82, 0x84, 0x80, 0x8a, 0xa7,
	0x83, 0xb0, 0xb2, 0xab, 0x31, 0x07, 0xa3, 0x02, 0x10, 0x23, 0x8b, 0xb2, 0x2b, 0x0a, 0xa7, 0xa3,
	0x02, 0x7b, 0x12, 0xc3, 0x07, 0x0c, 0x43, 0xa6, 0x91, 0x91, 0x9b, 0xaa, 0x82, 0xca, 0x2e, 0x6a,
	0x43, 0x51, 0x02, 0xcb, 0x52, 0x8b, 0x56, 0x57, 0x88, 0xc3, 0x83, 0x1a, 0x8d, 0x51, 0x86, 0x0a,
	0xc1, 0x1b, 0x22, 0x5a, 0x07, 0x84, 0xa3, 0xce, 0xba, 0xfa, 0xab, 0x6a, 0xea, 0x2c, 0x2e, 0x67,
	0x00, 0x33, 0x53, 0xd3, 0x47, 0x98, 0x93, 0x62, 0x2b, 0x9b, 0x2b, 0x82, 0xed, 0x4b, 0x1a, 0x86,
	0xa0, 0xb9, 0x82, 0x0b, 0x27, 0x09, 0xa2, 0xab, 0x20, 0x3a, 0x8b, 0x0a, 0x84, 0x8d, 0x0b, 0x8f,
	0x83, 0x8a, 0x92, 0x13, 0x10, 0x18, 0x06, 0x96, 0x83, 0x89, 0x8b, 0x92, 0x1c, 0x92, 0x9b, 0x17,
	0x02, 0x2b, 0x02, 0x02, 0x06, 0x25, 0xa2, 0xab, 0xa8, 0x12, 0x13, 0x9a, 0x21, 0x27, 0x03, 0x2a,
	0xa3, 0x92, 0x33, 0xb2, 0x94, 0x12, 0x32, 0x9b, 0x90, 0xa0, 0x8a, 0x2a, 0x9a, 0xbb, 0xae, 0x1e,
	0x41, 0x2b, 0x92, 0xb2, 0x44, 0xe0, 0x02, 0x6f, 0x61, 0x30, 0x4a, 0x13, 0x61, 0x4f, 0x2e, 0xa6,
	0x52, 0x00, 0xc2, 0x8b, 0x53, 0x8f, 0x93, 0x4f, 0x5b, 0x01, 0x1a, 0x9b, 0xc6, 0x01, 0x03, 0x0b,
	0x42, 0x09, 0xf2, 0x62, 0x82, 0x41, 0x22, 0xc6, 0x90, 0x2a, 0xfa, 0x0b, 0x6c, 0xa0, 0x4f, 0x03,
	0xa0, 0x53, 0xf2, 0xbb, 0x46, 0x96, 0x23, 0x22, 0xd8, 0xfa, 0x12, 0xab, 0x88, 0x1a, 0x7a, 0x8a,
};

const uint8_t cjddzsp_key[0x100] = {
	0x11, 0x21, 0xa2, 0x1a, 0x84, 0xaf, 0x26, 0x0b, 0x3b, 0xbb, 0x12, 0x9b, 0x89, 0x80, 0x2f, 0x0a,
	0x91, 0x80, 0x93, 0x93, 0x80, 0x0b, 0x13, 0x93, 0x0a, 0x82, 0x8a, 0x12, 0x13, 0x05, 0x96, 0x17,
	0x81, 0xb1, 0xb3, 0xab, 0x06, 0x2a, 0x87, 0x83, 0x33, 0x93, 0x13, 0x8a, 0x28, 0xa8, 0x07, 0x8b,
	0x11, 0xa3, 0xb2, 0xa2, 0x23, 0x17, 0x17, 0xb6, 0x33, 0xa9, 0xa3, 0x23, 0xa0, 0xa3, 0x9b, 0xbb,
	0x70, 0xe8, 0x83, 0x72, 0xe6, 0xa2, 0xa2, 0x27, 0xbb, 0xc8, 0xf3, 0x42, 0x6d, 0xc8, 0x66, 0x47,
	0x93, 0x18, 0x12, 0x12, 0x13, 0x58, 0xd2, 0xc6, 0x49, 0x09, 0xc3, 0x0a, 0x81, 0x0b, 0xc2, 0xda,
	0xd2, 0x33, 0xc2, 0x1a, 0x40, 0x89, 0x26, 0xeb, 0x78, 0x51, 0x5a, 0x62, 0xa3, 0xee, 0x02, 0x8f,
	0x42, 0xa1, 0xe3, 0x3a, 0x41, 0x44, 0x93, 0xd3, 0x03, 0xda, 0xe2, 0x83, 0x69, 0xc5, 0xb3, 0xb6,
	0x91, 0x00, 0xa2, 0x32, 0x24, 0x88, 0x87, 0xab, 0x02, 0x28, 0x2a, 0x8b, 0x87, 0xab, 0x2b, 0x8b,
	0x13, 0x02, 0x03, 0x9a, 0x94, 0x13, 0x87, 0x0b, 0x1a, 0x98, 0x03, 0x1b, 0x10, 0x81, 0x1a, 0x9f,
	0x81, 0xa9, 0x03, 0x3a, 0x05, 0x06, 0x27, 0xab, 0x3b, 0xa8, 0x8a, 0xab, 0xaf, 0x0a, 0xaa, 0x2f,
	0x31, 0x39, 0x32, 0x3a, 0x81, 0xbf, 0x07, 0x87, 0x89, 0x98, 0xa2, 0x22, 0x13, 0xa4, 0xb6, 0x0e,
	0x43, 0xf2, 0x43, 0x33, 0x47, 0x4c, 0x66, 0x26, 0xf2, 0x69, 0x2b, 0x5a, 0xa3, 0x83, 0x4b, 0xe6,
	0x41, 0x50, 0x92, 0xcb, 0xd3, 0x1e, 0x57, 0x87, 0x01, 0x19, 0x9a, 0x52, 0x45, 0x5a, 0x9e, 0xde,
	0xa3, 0xa1, 0x42, 0x7b, 0xa3, 0x22, 0xa2, 0x87, 0x80, 0xe0, 0xf3, 0x23, 0x2a, 0x8e, 0x2f, 0x6f,
	0x92, 0x1a, 0x23, 0xab, 0xb3, 0x09, 0xd6, 0xab, 0x38, 0xe3, 0x2b, 0x3a, 0xdf, 0x7d, 0xea, 0x87,
};

const uint8_t cjdh2_key[0x100] = {
	0x03, 0x31, 0x92, 0x23, 0x21, 0x2b, 0x23, 0x23, 0x39, 0x01, 0xb2, 0x9b, 0x0d, 0xaa, 0x07, 0x86,
	0x03, 0x9b, 0x03, 0x82, 0x82, 0x00, 0x86, 0x0b, 0x80, 0x92, 0x9a, 0x1b, 0x81, 0x9a, 0x92, 0x8f,
	0x83, 0x89, 0x82, 0x0a, 0x02, 0x0f, 0x83, 0xa7, 0x80, 0x32, 0xbb, 0x02, 0x8f, 0xa2, 0xaa, 0x0e,
	0x80, 0x12, 0x23, 0xbb, 0x86, 0xb9, 0xb3, 0x1b, 0x19, 0xb8, 0x93, 0x22, 0x28, 0x9d, 0xbf, 0xb2,
	0xa1, 0xb0, 0x63, 0xaa, 0x81, 0x8a, 0x47, 0x0b, 0xdb, 0x21, 0x5a, 0x03, 0xe9, 0x60, 0x2f, 0xab,
	0x00, 0x43, 0xc2, 0x8b, 0x06, 0x54, 0x47, 0x9f, 0x51, 0xc9, 0x4a, 0x4b, 0x1f, 0x40, 0x9f, 0x52,
	0x21, 0x00, 0xe3, 0x72, 0x44, 0x43, 0xc2, 0xab, 0x5a, 0x32, 0x1a, 0x62, 0x6d, 0xa2, 0x82, 0xce,
	0x73, 0xe0, 0xc3, 0xa3, 0x73, 0x71, 0x16, 0x42, 0x69, 0xc9, 0x02, 0x43, 0x93, 0x23, 0x43, 0xbf,
	0x83, 0x19, 0xb2, 0x9a, 0xa0, 0x8a, 0x03, 0x8e, 0x29, 0x03, 0x02, 0x0b, 0xa0, 0xa0, 0x8b, 0x0a,
	0x13, 0x0b, 0x12, 0x9a, 0x10, 0x80, 0x87, 0x8f, 0x98, 0x89, 0x13, 0x0b, 0x83, 0x8e, 0x1a, 0x1a,
	0x90, 0xab, 0xa2, 0x9b, 0xa5, 0xae, 0x22, 0x0a, 0x8b, 0xab, 0xa3, 0x0a, 0x0e, 0x02, 0x8e, 0x0f,
	0x32, 0x3b, 0x13, 0x0b, 0x93, 0x91, 0x22, 0x0b, 0x90, 0xab, 0xb2, 0x33, 0xa1, 0x21, 0xaa, 0xae,
	0xa3, 0x93, 0x73, 0xc2, 0x67, 0x81, 0xc7, 0x0a, 0x31, 0xa2, 0x7b, 0x93, 0xa7, 0x60, 0x86, 0xce,
	0x53, 0x18, 0x53, 0x52, 0xc6, 0x5b, 0x47, 0x1a, 0x0b, 0x98, 0x5b, 0xda, 0x92, 0x14, 0x07, 0x82,
	0x70, 0xc3, 0x02, 0xd2, 0xe1, 0x42, 0x42, 0x47, 0xe3, 0x20, 0x9a, 0xea, 0xe6, 0x02, 0x2a, 0x8f,
	0xf3, 0x3a, 0x22, 0x7a, 0xf1, 0x58, 0x97, 0xeb, 0x41, 0x59, 0xe2, 0x73, 0xdd, 0xa7, 0x7e, 0x1f,
};

const uint8_t bubucar_key[0x100] = {
	0x01, 0x91, 0x31, 0x00, 0x83, 0x2e, 0x05, 0x0d, 0x02, 0x31, 0x91, 0x18, 0x21, 0x2e, 0xac, 0xad,
	0x93, 0x8b, 0x91, 0x11, 0x07, 0x99, 0x84, 0x0c, 0x8b, 0x03, 0x91, 0x01, 0x84, 0x94, 0x88, 0x95,
	0x10, 0x9b, 0x91, 0xb0, 0x87, 0x80, 0x04, 0x8d, 0x82, 0xb0, 0xb0, 0x30, 0x87, 0x25, 0x80, 0x29,
	0x91, 0x10, 0x01, 0x89, 0x27, 0x8a, 0x14, 0x31, 0xba, 0xba, 0x20, 0x91, 0xbc, 0x3b, 0x3c, 0xa1,
	0x72, 0x38, 0x10, 0xe1, 0xe2, 0x88, 0xa0, 0xed, 0xb9, 0xe8, 0xf0, 0x51, 0xa6, 0x6e, 0x40, 0x65,
	0x50, 0xc8, 0x80, 0x00, 0x41, 0xc7, 0x44, 0x9d, 0x80, 0x9b, 0x00, 0x99, 0x8c, 0x81, 0x10, 0x08,
	0x61, 0x61, 0x00, 0x39, 0xa6, 0xc7, 0x85, 0x25, 0x80, 0xb2, 0xb9, 0x10, 0xa2, 0x6a, 0xc1, 0x49,
	0xd1, 0xf0, 0xe0, 0xc9, 0x82, 0x87, 0xc4, 0xec, 0xf2, 0x90, 0x29, 0xe1, 0x5d, 0x6c, 0x41, 0x80,
	0x32, 0xaa, 0x80, 0x00, 0xa6, 0x26, 0x00, 0xa9, 0x82, 0x92, 0x30, 0xb1, 0x02, 0xa0, 0x84, 0xa5,
	0x80, 0x93, 0x90, 0x08, 0x06, 0x03, 0x15, 0x94, 0x0b, 0x83, 0x91, 0x10, 0x97, 0x17, 0x81, 0x81,
	0xa0, 0xa9, 0x10, 0xa8, 0xa7, 0x2d, 0x81, 0x05, 0x98, 0x9a, 0x88, 0x80, 0xaf, 0x81, 0x29, 0xad,
	0x20, 0xb9, 0xb0, 0xa9, 0xa4, 0x33, 0x31, 0x90, 0x90, 0x18, 0x88, 0x90, 0x21, 0xaf, 0x2c, 0x80,
	0x50, 0x58, 0x80, 0xb1, 0x20, 0x64, 0xe0, 0x2c, 0xc2, 0x48, 0xe1, 0x40, 0x60, 0x23, 0xc4, 0xe8,
	0x42, 0x8b, 0x50, 0x48, 0xd4, 0x0f, 0x00, 0xd0, 0x98, 0x41, 0xc0, 0x89, 0xc3, 0x0e, 0x9c, 0x1d,
	0xe1, 0xb2, 0xc0, 0x80, 0x43, 0x42, 0x41, 0xa8, 0x5b, 0xa3, 0x68, 0x40, 0xae, 0x60, 0x01, 0x24,
	0x31, 0xb3, 0xf0, 0xf8, 0x34, 0x8e, 0xc0, 0x94, 0xa8, 0xd8, 0x10, 0xa0, 0x46, 0x33, 0xe9, 0x38,
};

const uint8_t qhzb_key[0x100] = {
	0x20, 0xba, 0xb2, 0x93, 0xa5, 0xa6, 0xa2, 0x83, 0xaa, 0x89, 0x12, 0xaa, 0x82, 0xa2, 0x8a, 0x8f,
	0x90, 0x99, 0x93, 0x93, 0x84, 0x98, 0x97, 0x92, 0x82, 0x9b, 0x93, 0x0a, 0x9e, 0x0b, 0x0e, 0x92,
	0xa1, 0xb9, 0x93, 0xa3, 0x01, 0xab, 0x06, 0x23, 0x80, 0x93, 0x13, 0x92, 0xa5, 0xa0, 0x2a, 0x2b,
	0x80, 0x32, 0x82, 0xa3, 0x32, 0x0e, 0xb2, 0xaf, 0x80, 0x29, 0x8b, 0x3a, 0x3c, 0x1d, 0xa6, 0x0f,
	0xc0, 0x19, 0x22, 0x7a, 0x07, 0x42, 0xa2, 0xe6, 0x42, 0xfb, 0x12, 0xca, 0x0a, 0x45, 0x4b, 0x43,
	0xd2, 0x9b, 0xc3, 0x4b, 0x55, 0xcb, 0x42, 0x06, 0x89, 0x12, 0xd2, 0xca, 0x86, 0xc4, 0x52, 0xd2,
	0x03, 0x49, 0x32, 0x7a, 0xa2, 0x8c, 0x22, 0x03, 0x88, 0xda, 0xc2, 0x2a, 0x86, 0x03, 0x66, 0x6e,
	0x43, 0xa0, 0xb2, 0x2b, 0xb7, 0xb7, 0x33, 0x8f, 0x02, 0xb8, 0x83, 0xb3, 0x1b, 0x0f, 0x7f, 0x6b,
	0xa0, 0xb1, 0x12, 0x82, 0x07, 0x20, 0x02, 0x8b, 0x82, 0x1a, 0xbb, 0x93, 0xa2, 0x2a, 0xa6, 0x0e,
	0x82, 0x81, 0x13, 0x12, 0x15, 0x9a, 0x06, 0x83, 0x1a, 0x89, 0x93, 0x8a, 0x9c, 0x90, 0x03, 0x8e,
	0x30, 0x19, 0x02, 0x8a, 0xa6, 0x28, 0xa2, 0x0b, 0x2a, 0x81, 0xba, 0x1a, 0x86, 0xaa, 0x82, 0x22,
	0xa1, 0x39, 0xb2, 0x83, 0xb2, 0xbc, 0x27, 0x33, 0x39, 0x9b, 0x92, 0xb2, 0x21, 0x05, 0x96, 0x3b,
	0xf3, 0xe9, 0x13, 0xf3, 0xc6, 0xea, 0xe6, 0xe7, 0x82, 0x71, 0x0b, 0x52, 0x8a, 0x6b, 0x8f, 0x0b,
	0x11, 0x82, 0x53, 0x4a, 0x54, 0x95, 0x02, 0xc2, 0x80, 0x90, 0x02, 0x52, 0xc9, 0x0f, 0x1f, 0xcf,
	0xc0, 0xe8, 0x13, 0xa3, 0x24, 0x0f, 0x03, 0x66, 0x92, 0x38, 0xb3, 0x9b, 0xa5, 0x4d, 0xa7, 0x23,
	0xb1, 0x0b, 0x83, 0xab, 0x20, 0xad, 0x36, 0x5f, 0xc2, 0x3a, 0x3b, 0x33, 0x3d, 0xcf, 0xea, 0x22,
};

const uint8_t lhtb_key[0x100] = {
	0x92, 0x93, 0x83, 0x02, 0xa4, 0x8f, 0xa6, 0xa7, 0x3a, 0xa2, 0xba, 0x0a, 0x0e, 0x01, 0x8f, 0x2f,
	0x83, 0x81, 0x13, 0x1b, 0x05, 0x87, 0x83, 0x87, 0x8a, 0x9b, 0x83, 0x1a, 0x8b, 0x93, 0x96, 0x12,
	0x12, 0xb9, 0x12, 0x0a, 0x81, 0x8d, 0x03, 0xa7, 0x30, 0x29, 0xa2, 0x82, 0x8e, 0x22, 0x87, 0x26,
	0x91, 0x08, 0x12, 0x82, 0x12, 0x1a, 0x26, 0x0e, 0x10, 0x18, 0x3a, 0x03, 0x94, 0x0b, 0xa7, 0x9f,
	0xc0, 0x7b, 0xc3, 0x1a, 0x03, 0x67, 0x07, 0xcb, 0x43, 0xaa, 0xfa, 0x8b, 0xe5, 0x6b, 0x86, 0x62,
	0x00, 0x8a, 0x82, 0xdb, 0xc7, 0x91, 0xd7, 0x52, 0xdb, 0x08, 0xd3, 0xdb, 0x83, 0xcd, 0xcf, 0x92,
	0x33, 0x48, 0xa2, 0x02, 0x25, 0x81, 0x62, 0xc7, 0x53, 0xe9, 0x8b, 0x13, 0xe7, 0x2b, 0x02, 0x23,
	0xe1, 0xb1, 0x42, 0xc2, 0xa4, 0x63, 0xf6, 0x2b, 0xba, 0x6a, 0xc3, 0x32, 0x48, 0xb0, 0xfe, 0xe3,
	0x33, 0x39, 0x82, 0x1a, 0x27, 0x82, 0x26, 0x87, 0x98, 0xb3, 0x12, 0x12, 0x8e, 0x87, 0x86, 0xab,
	0x12, 0x0a, 0x12, 0x12, 0x11, 0x09, 0x83, 0x97, 0x12, 0x82, 0x9a, 0x12, 0x83, 0x9c, 0x12, 0x8a,
	0x81, 0xa9, 0x32, 0xba, 0x00, 0x82, 0x83, 0xaf, 0x12, 0x00, 0x9a, 0x2a, 0x86, 0x29, 0x07, 0x22,
	0x31, 0x3b, 0xa3, 0x13, 0xb1, 0x3a, 0x27, 0x36, 0x38, 0x99, 0x3b, 0x03, 0x04, 0x1a, 0x23, 0x3a,
	0xc3, 0xe1, 0x32, 0x1b, 0x62, 0x2a, 0x43, 0xae, 0x30, 0x53, 0xab, 0x4b, 0x05, 0x89, 0x83, 0xc6,
	0x81, 0x90, 0x53, 0x92, 0xd2, 0x8e, 0x03, 0xd6, 0xc1, 0x1a, 0x53, 0xc3, 0xcb, 0xc1, 0xc7, 0xd2,
	0xb3, 0x51, 0xe2, 0xb3, 0x87, 0x61, 0x26, 0x47, 0x68, 0x10, 0x2a, 0x2b, 0x2e, 0xcd, 0xee, 0x67,
	0x62, 0x28, 0x03, 0x7a, 0x72, 0x1e, 0xb2, 0x5e, 0x40, 0x51, 0x22, 0x82, 0x4b, 0xd9, 0x56, 0xe6,
};

const uint8_t lhzb3in1_key[0x100] = {
	0xb1, 0x00, 0xa3, 0xab, 0xa4, 0x22, 0x83, 0x8b, 0xab, 0xa1, 0x1a, 0x22, 0x09, 0x8d, 0x03, 0xae,
	0x03, 0x08, 0x82, 0x9a, 0x05, 0x89, 0x07, 0x0b, 0x11, 0x1b, 0x13, 0x03, 0x96, 0x18, 0x12, 0x87,
	0x03, 0xb8, 0x32, 0x0b, 0xa1, 0x21, 0x23, 0x2e, 0x01, 0x92, 0x8b, 0x83, 0x2f, 0x00, 0xa7, 0x23,
	0xb2, 0x92, 0xb2, 0x9a, 0xb4, 0xbf, 0x27, 0x93, 0xa8, 0x11, 0x23, 0x0b, 0xb1, 0x84, 0x23, 0x2e,
	0x92, 0xe0, 0x73, 0x1b, 0x40, 0x8f, 0x62, 0xa3, 0xbb, 0x02, 0x42, 0xe2, 0x89, 0x2f, 0xce, 0xae,
	0x12, 0x9a, 0xd2, 0x8a, 0x52, 0x8b, 0x93, 0x4b, 0xd0, 0x0a, 0xd3, 0x5b, 0xcc, 0x1b, 0xc7, 0xca,
	0xe1, 0x7b, 0x43, 0xb3, 0xa2, 0xc7, 0xe7, 0xc6, 0x7b, 0xf2, 0x23, 0x3a, 0x2b, 0xec, 0xce, 0x43,
	0x53, 0x69, 0xd3, 0xcb, 0x91, 0xd5, 0x02, 0xf6, 0x42, 0x13, 0x13, 0xd2, 0x47, 0xf5, 0xa2, 0xf6,
	0x22, 0x20, 0xa3, 0xa3, 0x21, 0x06, 0x26, 0x26, 0x00, 0x02, 0x12, 0x8b, 0x2d, 0x81, 0x2a, 0xae,
	0x81, 0x82, 0x93, 0x93, 0x12, 0x93, 0x16, 0x12, 0x89, 0x10, 0x93, 0x02, 0x92, 0x8b, 0x07, 0x0a,
	0x93, 0x9b, 0x83, 0x02, 0xa6, 0x00, 0x26, 0x2b, 0x82, 0x19, 0xab, 0xab, 0x87, 0x2e, 0xae, 0x0e,
	0xb0, 0xb2, 0x93, 0x1b, 0xb5, 0x1e, 0xa3, 0x16, 0x08, 0x30, 0xb3, 0xab, 0xbd, 0x34, 0x2f, 0xbe,
	0x51, 0x71, 0xe2, 0xb2, 0x41, 0xe4, 0x66, 0x87, 0xf2, 0xa8, 0xd2, 0xfa, 0x2e, 0x2a, 0xe2, 0xeb,
	0xd0, 0x80, 0xd3, 0xd3, 0x92, 0x01, 0xc7, 0x46, 0x59, 0x5b, 0x4b, 0x12, 0x09, 0x89, 0x8a, 0x86,
	0xc0, 0x5a, 0xc3, 0xfa, 0x85, 0xcb, 0xc7, 0x46, 0xea, 0x92, 0x33, 0x9a, 0x21, 0x8d, 0xcb, 0x0b,
	0x90, 0xf1, 0x32, 0x7b, 0x40, 0xce, 0x06, 0x72, 0x88, 0xf2, 0x33, 0xab, 0xe1, 0x59, 0x42, 0x1e,
};

const uint8_t jxry_key[0x100] = {
	0xa0, 0x9a, 0x33, 0x1a, 0x83, 0x8a, 0x26, 0x27, 0x8a, 0x00, 0x13, 0x83, 0x84, 0x85, 0x0a, 0x8b,
	0x92, 0x8a, 0x13, 0x8b, 0x02, 0x19, 0x93, 0x92, 0x91, 0x13, 0x82, 0x02, 0x8f, 0x17, 0x07, 0x13,
	0x93, 0xb8, 0xb2, 0x92, 0x26, 0x2d, 0x87, 0x8a, 0x21, 0x00, 0x1b, 0xa3, 0x2d, 0x00, 0x2b, 0x23,
	0x23, 0x3a, 0x03, 0xa3, 0x21, 0x08, 0xb2, 0x2f, 0x22, 0x0a, 0xa2, 0x23, 0x96, 0x8b, 0x22, 0xa6,
	0xa3, 0xd3, 0xb3, 0x43, 0x04, 0xe0, 0x82, 0xe7, 0x38, 0x10, 0x22, 0x43, 0x46, 0xad, 0xce, 0x67,
	0x91, 0x02, 0xd2, 0x4b, 0xc4, 0xc9, 0x56, 0x1e, 0x19, 0xda, 0x13, 0x8a, 0x94, 0x02, 0x5a, 0xd7,
	0xf3, 0x09, 0xf2, 0x12, 0x21, 0x09, 0x06, 0x2e, 0x51, 0xa2, 0x62, 0xc3, 0x82, 0x09, 0x8f, 0x47,
	0xd2, 0x21, 0x13, 0xaa, 0x42, 0xb4, 0xf7, 0x37, 0x5a, 0x1b, 0x02, 0x5b, 0x28, 0x55, 0xc7, 0x42,
	0xb0, 0x03, 0xa3, 0xa2, 0xa6, 0x03, 0x02, 0x0e, 0xb0, 0x81, 0xa2, 0x9b, 0x29, 0xad, 0x0a, 0x0b,
	0x90, 0x8b, 0x13, 0x0a, 0x83, 0x90, 0x17, 0x13, 0x88, 0x1a, 0x13, 0x9b, 0x94, 0x1c, 0x03, 0x17,
	0x91, 0x10, 0x13, 0xa2, 0xa2, 0x2f, 0x07, 0xab, 0x98, 0x12, 0x2b, 0x22, 0x8a, 0x00, 0x26, 0x2b,
	0x02, 0x39, 0xa3, 0x33, 0x20, 0x15, 0x37, 0x36, 0x19, 0x23, 0xb3, 0xbb, 0xa6, 0xb6, 0x23, 0x8a,
	0x72, 0x10, 0x23, 0x4a, 0xe4, 0xc4, 0x07, 0x0b, 0xba, 0x38, 0x02, 0xb2, 0xc7, 0x43, 0x4b, 0xcf,
	0x90, 0x82, 0x83, 0x4b, 0x97, 0x5c, 0xd6, 0xce, 0x83, 0xd8, 0x0b, 0xdb, 0x8f, 0x4c, 0x1e, 0x0b,
	0x63, 0x58, 0x43, 0x9a, 0x27, 0x47, 0xa3, 0x6e, 0x5a, 0x2b, 0xb3, 0x5b, 0x46, 0xce, 0x82, 0x6f,
	0x12, 0x03, 0x83, 0xf2, 0x62, 0x2d, 0x92, 0xaf, 0xfb, 0x08, 0x7b, 0x83, 0xf6, 0xa6, 0x53, 0x97,
};

const uint8_t dahuaxy_key[0x100] = {
	0x00, 0x2b, 0x92, 0x0b, 0x00, 0x2f, 0x26, 0xa6, 0xbb, 0x92, 0xbb, 0x22, 0x01, 0x2b, 0x83, 0xa7,
	0x81, 0x01, 0x82, 0x93, 0x86, 0x88, 0x06, 0x07, 0x1b, 0x11, 0x02, 0x83, 0x05, 0x04, 0x82, 0x86,
	0xa0, 0xb2, 0x13, 0x92, 0x80, 0xa2, 0x03, 0x2a, 0xbb, 0x2b, 0x2a, 0x93, 0x8c, 0x85, 0x26, 0x03,
	0x82, 0xa2, 0x22, 0x02, 0x90, 0x1d, 0x22, 0xaf, 0x08, 0x93, 0x22, 0x92, 0x38, 0xbb, 0x93, 0x83,
	0x13, 0xb1, 0x93, 0x92, 0x03, 0x60, 0xe7, 0x2f, 0x01, 0x5b, 0x13, 0x52, 0xa5, 0x85, 0x47, 0xe6,
	0x01, 0x59, 0xd2, 0x93, 0x86, 0xd2, 0xc7, 0x02, 0x11, 0x8a, 0x4a, 0x43, 0x50, 0x0a, 0x8f, 0x5a,
	0x50, 0xaa, 0x82, 0x93, 0x85, 0x0b, 0x47, 0x22, 0xdb, 0x6b, 0x23, 0x12, 0xa7, 0x28, 0xcb, 0xee,
	0x82, 0xd9, 0x53, 0x22, 0x26, 0xa1, 0x12, 0x3a, 0x7a, 0x08, 0x83, 0x7b, 0x96, 0x67, 0x9f, 0x33,
	0xa2, 0x00, 0x13, 0x8b, 0x26, 0x0b, 0x23, 0x2a, 0x91, 0x2a, 0x3b, 0x83, 0x8a, 0x0f, 0xa6, 0x02,
	0x00, 0x9b, 0x13, 0x82, 0x91, 0x01, 0x12, 0x1e, 0x8b, 0x9b, 0x8b, 0x83, 0x10, 0x9c, 0x0e, 0x9e,
	0x93, 0xb8, 0x93, 0xba, 0x26, 0xa9, 0x22, 0xa7, 0x8b, 0x23, 0x12, 0xa2, 0xa6, 0x20, 0x03, 0xab,
	0xa1, 0x3b, 0x93, 0x12, 0x30, 0x2a, 0x36, 0x23, 0xab, 0x03, 0x23, 0x9b, 0x08, 0x81, 0x03, 0x83,
	0x61, 0xea, 0xf3, 0x4b, 0x85, 0x6b, 0x82, 0x02, 0x90, 0xd3, 0xdb, 0x3a, 0xc3, 0xa5, 0xea, 0xe3,
	0x13, 0x81, 0x12, 0x02, 0x02, 0x54, 0xd7, 0x02, 0x88, 0x0a, 0xd3, 0x1a, 0x84, 0xc4, 0x46, 0x8a,
	0x12, 0x62, 0xa2, 0xfb, 0xe3, 0xcb, 0x42, 0x62, 0x8a, 0x83, 0xcb, 0xdb, 0x49, 0xed, 0xc2, 0x4b,
	0xe0, 0xd9, 0xc3, 0x13, 0x66, 0x38, 0xd6, 0x67, 0xd9, 0xba, 0xdb, 0x3b, 0x0e, 0xe6, 0x5e, 0x33,
};

const uint8_t slqzsp_key[0x100] = {
	0x20, 0xba, 0xb2, 0x93, 0xa5, 0xa6, 0xa2, 0x83, 0xaa, 0x89, 0x12, 0xaa, 0x82, 0xa2, 0x8a, 0x8f,
	0x90, 0x99, 0x93, 0x93, 0x84, 0x98, 0x97, 0x92, 0x82, 0x9b, 0x93, 0x0a, 0x9e, 0x0b, 0x0e, 0x92,
	0xa1, 0xb9, 0x93, 0xa3, 0x01, 0xab, 0x06, 0x23, 0x80, 0x93, 0x13, 0x92, 0xa5, 0xa0, 0x2a, 0x2b,
	0x80, 0x32, 0x82, 0xa3, 0x32, 0x0e, 0xb2, 0xaf, 0x80, 0x29, 0x8b, 0x3a, 0x3c, 0x1d, 0xa6, 0x0f,
	0xc0, 0x19, 0x22, 0x7a, 0x07, 0x42, 0xa2, 0xe6, 0x42, 0xfb, 0x12, 0xca, 0x0a, 0x45, 0x4b, 0x43,
	0xd2, 0x9b, 0xc3, 0x4b, 0x55, 0xcb, 0x42, 0x06, 0x89, 0x12, 0xd2, 0xca, 0x86, 0xc4, 0x52, 0xd2,
	0x03, 0x49, 0x32, 0x7a, 0xa2, 0x8c, 0x22, 0x03, 0x88, 0xda, 0xc2, 0x2a, 0x86, 0x03, 0x66, 0x6e,
	0x43, 0xa0, 0xb2, 0x2b, 0xb7, 0xb7, 0x33, 0x8f, 0x02, 0xb8, 0x83, 0xb3, 0x1b, 0x0f, 0x7f, 0x6b,
	0xa0, 0xb1, 0x12, 0x82, 0x07, 0x20, 0x02, 0x8b, 0x82, 0x1a, 0xbb, 0x93, 0xa2, 0x2a, 0xa6, 0x0e,
	0x82, 0x81, 0x13, 0x12, 0x15, 0x9a, 0x06, 0x83, 0x1a, 0x89, 0x93, 0x8a, 0x9c, 0x90, 0x03, 0x8e,
	0x30, 0x19, 0x02, 0x8a, 0xa6, 0x28, 0xa2, 0x0b, 0x2a, 0x81, 0xba, 0x1a, 0x86, 0xaa, 0x82, 0x22,
	0xa1, 0x39, 0xb2, 0x83, 0xb2, 0xbc, 0x27, 0x33, 0x39, 0x9b, 0x92, 0xb2, 0x21, 0x05, 0x96, 0x3b,
	0xf3, 0xe9, 0x13, 0xf3, 0xc6, 0xea, 0xe6, 0xe7, 0x82, 0x71, 0x0b, 0x52, 0x8a, 0x6b, 0x8f, 0x0b,
	0x11, 0x82, 0x53, 0x4a, 0x54, 0x95, 0x02, 0xc2, 0x80, 0x90, 0x02, 0x52, 0xc9, 0x0f, 0x1f, 0xcf,
	0xc0, 0xe8, 0x13, 0xa3, 0x24, 0x0f, 0x03, 0x66, 0x92, 0x38, 0xb3, 0x9b, 0xa5, 0x4d, 0xa7, 0x23,
	0xb1, 0x0b, 0x83, 0xab, 0x20, 0xad, 0x36, 0x5f, 0xc2, 0x3a, 0x3b, 0x33, 0x3d, 0xcf, 0xea, 0x22,
};

const uint8_t sydh_key[0x100] = {
	0xb3, 0xb3, 0xa3, 0x93, 0x26, 0x2e, 0x83, 0xa6, 0x21, 0x29, 0xba, 0xab, 0x80, 0x85, 0x0e, 0x0e,
	0x93, 0x0b, 0x93, 0x9a, 0x92, 0x13, 0x86, 0x82, 0x0a, 0x9b, 0x93, 0x8b, 0x1f, 0x93, 0x0f, 0x1f,
	0x31, 0x18, 0x32, 0x22, 0xa5, 0x87, 0xa2, 0x87, 0xba, 0x99, 0xab, 0xba, 0x09, 0xa1, 0xa2, 0x2e,
	0x13, 0x02, 0xb3, 0xba, 0x15, 0x3e, 0xb6, 0xb3, 0xb9, 0x18, 0x23, 0x22, 0x2c, 0x9a, 0x92, 0x3b,
	0xc1, 0x6a, 0x73, 0x3b, 0x61, 0xea, 0xc6, 0x8f, 0x08, 0x81, 0xb3, 0x73, 0x2e, 0x60, 0x0e, 0x0a,
	0x51, 0xd1, 0xc2, 0x5a, 0x95, 0x5b, 0x07, 0xc3, 0x10, 0x82, 0xca, 0xc3, 0xd3, 0x90, 0x0a, 0x57,
	0xe0, 0x58, 0xc3, 0x53, 0xa6, 0x6b, 0x03, 0x82, 0xf2, 0xc0, 0xbb, 0xf2, 0x01, 0x2b, 0xcf, 0x66,
	0xc2, 0x81, 0xc2, 0x9a, 0x75, 0x7d, 0x33, 0x5f, 0xda, 0x43, 0x32, 0x3b, 0x7b, 0xaa, 0xd7, 0x9f,
	0x21, 0xab, 0xb3, 0xab, 0x83, 0x2c, 0xa6, 0x82, 0xaa, 0x21, 0x0b, 0x12, 0xa9, 0x02, 0xab, 0xaa,
	0x82, 0x18, 0x03, 0x83, 0x15, 0x9d, 0x97, 0x9e, 0x0b, 0x9a, 0x9b, 0x92, 0x0d, 0x83, 0x0f, 0x1b,
	0x11, 0x0a, 0xb3, 0x9b, 0x81, 0x2b, 0xa2, 0x03, 0x13, 0x10, 0x8a, 0x92, 0x0e, 0xa7, 0x8f, 0xab,
	0xa0, 0xa0, 0x23, 0xa2, 0x12, 0x35, 0xa2, 0xab, 0x38, 0x83, 0x1a, 0x8a, 0x11, 0x0f, 0x87, 0x8e,
	0xe3, 0x39, 0x92, 0x9b, 0x83, 0x68, 0xe7, 0x4b, 0x12, 0x11, 0x92, 0x3b, 0xa9, 0x08, 0xa2, 0xa2,
	0xc1, 0x01, 0x43, 0x5b, 0xd3, 0x5b, 0x96, 0x5a, 0x91, 0xd3, 0xcb, 0xd2, 0x8e, 0x87, 0x17, 0x87,
	0x90, 0xe9, 0x22, 0xfa, 0x61, 0x85, 0x83, 0x83, 0xd2, 0x89, 0x32, 0xd3, 0xe2, 0x04, 0x6a, 0x87,
	0x80, 0xe1, 0xb3, 0x12, 0xe0, 0x27, 0x03, 0x2b, 0xc0, 0xd9, 0xa3, 0xaa, 0xec, 0x32, 0x1e, 0xb6,
};

const uint8_t xyddz_key[0x100] = {
	0x00, 0x89, 0x12, 0x2b, 0xa0, 0xac, 0xa3, 0xae, 0x29, 0xb3, 0xaa, 0x22, 0x05, 0xa5, 0xab, 0x86,
	0x80, 0x0a, 0x82, 0x03, 0x04, 0x05, 0x93, 0x16, 0x8a, 0x92, 0x9b, 0x13, 0x84, 0x05, 0x02, 0x8e,
	0x22, 0x28, 0x22, 0x33, 0x02, 0x06, 0x27, 0x07, 0x2b, 0x81, 0x22, 0x33, 0x88, 0x88, 0x82, 0x8b,
	0x91, 0xb8, 0x82, 0x92, 0x06, 0x04, 0x22, 0x2f, 0x1b, 0xb0, 0xaa, 0x9a, 0x1e, 0xac, 0xa6, 0xaa,
	0x21, 0x8a, 0xd3, 0x42, 0x06, 0xa9, 0xa2, 0xef, 0xe1, 0x00, 0xc3, 0x63, 0x48, 0xe6, 0xcb, 0x23,
	0x83, 0x10, 0x12, 0x8b, 0x15, 0x0d, 0xc3, 0x17, 0x93, 0xc9, 0x9b, 0x5b, 0x8e, 0x4b, 0x92, 0x56,
	0xb3, 0x79, 0x32, 0x4a, 0x22, 0xa2, 0xc2, 0x27, 0x28, 0x31, 0x43, 0xda, 0x63, 0x0c, 0x07, 0xc2,
	0x61, 0x91, 0x12, 0xa3, 0x01, 0x91, 0xa7, 0xa6, 0xa1, 0x99, 0x33, 0x43, 0x3d, 0x12, 0xd2, 0xbe,
	0xa3, 0x89, 0x13, 0x93, 0x27, 0x22, 0x82, 0x23, 0xa3, 0x22, 0x3a, 0x1a, 0x21, 0x2d, 0x26, 0x2f,
	0x10, 0x0b, 0x12, 0x8b, 0x81, 0x0e, 0x82, 0x1e, 0x91, 0x09, 0x1a, 0x8a, 0x8b, 0x08, 0x9e, 0x96,
	0x33, 0xb3, 0xb2, 0x0a, 0x24, 0x80, 0x02, 0x0e, 0x3b, 0x1b, 0x13, 0x9b, 0x28, 0x02, 0x06, 0xa2,
	0x83, 0x02, 0x93, 0xbb, 0xa6, 0x37, 0x26, 0x2e, 0xa8, 0x08, 0xa3, 0xaa, 0xb7, 0x92, 0x1e, 0x87,
	0xe3, 0x12, 0x02, 0x6b, 0x82, 0x8c, 0xa2, 0xc2, 0x12, 0xe2, 0x93, 0x93, 0x83, 0x8b, 0x4f, 0xef,
	0x82, 0x83, 0x03, 0xd3, 0x17, 0x41, 0x53, 0x92, 0xc9, 0xc2, 0x4b, 0x02, 0xca, 0x58, 0x56, 0x0a,
	0xa3, 0xba, 0x03, 0x2b, 0xa1, 0xef, 0xc7, 0xe7, 0x90, 0xc1, 0x1b, 0xfa, 0x68, 0x80, 0x83, 0x27,
	0x52, 0xb1, 0x43, 0xea, 0x95, 0xae, 0x67, 0x02, 0x49, 0xc8, 0x7b, 0x22, 0xc3, 0xed, 0x72, 0xe7,
};

const uint8_t cjbq_key[0x100] = {
	0x02, 0x83, 0x92, 0x3a, 0x07, 0x20, 0x86, 0x83, 0x13, 0xba, 0x2b, 0x03, 0x07, 0x01, 0x06, 0x87,
	0x00, 0x01, 0x12, 0x9b, 0x01, 0x87, 0x97, 0x1a, 0x1a, 0x0b, 0x8a, 0x13, 0x92, 0x8d, 0x1b, 0x92,
	0x21, 0x23, 0x23, 0x83, 0x01, 0x0c, 0x22, 0xae, 0x19, 0x89, 0x9b, 0x22, 0xa9, 0xae, 0xaa, 0x27,
	0x11, 0x31, 0x83, 0x93, 0x32, 0x96, 0xa3, 0x3b, 0xa2, 0x82, 0x1b, 0x13, 0x91, 0x8a, 0x82, 0x9f,
	0x33, 0x8a, 0xd3, 0xca, 0xc4, 0x00, 0x43, 0x82, 0x49, 0x5b, 0x43, 0xfb, 0x87, 0xa7, 0x06, 0x86,
	0x93, 0x83, 0x52, 0x8b, 0xd4, 0x99, 0x07, 0x4f, 0x81, 0x82, 0x8a, 0xdb, 0x9d, 0x8a, 0x9b, 0x5b,
	0x32, 0x38, 0x22, 0x3a, 0x02, 0xc9, 0xc3, 0x03, 0x98, 0x49, 0x62, 0xaa, 0x8f, 0x80, 0xc6, 0xa2,
	0x91, 0x53, 0x63, 0x4a, 0x95, 0x0e, 0x23, 0xe3, 0x02, 0x21, 0x4a, 0xea, 0xae, 0x54, 0x97, 0xef,
	0x32, 0x20, 0x92, 0x82, 0x24, 0xab, 0x07, 0xa3, 0x32, 0x88, 0x3a, 0x8a, 0x29, 0x2b, 0xa6, 0x02,
	0x80, 0x09, 0x83, 0x0b, 0x95, 0x87, 0x16, 0x9f, 0x12, 0x80, 0x1b, 0x13, 0x8e, 0x1e, 0x0a, 0x0f,
	0x82, 0x2a, 0x23, 0xbb, 0x21, 0xae, 0xa3, 0xa2, 0xba, 0x32, 0xb3, 0x33, 0x2c, 0x20, 0xae, 0x0a,
	0xb0, 0x19, 0x92, 0x12, 0xb6, 0x82, 0xa2, 0x86, 0x09, 0x18, 0x33, 0xab, 0x83, 0x92, 0xb2, 0xa2,
	0xe0, 0x32, 0xe3, 0x02, 0xc2, 0x2e, 0x63, 0xa2, 0x31, 0x98, 0x4a, 0x3a, 0x41, 0x05, 0xa7, 0xe7,
	0x80, 0xdb, 0xc2, 0x8b, 0xd6, 0x41, 0x06, 0xca, 0x43, 0xd3, 0x43, 0x13, 0x05, 0x89, 0x17, 0x1b,
	0xa2, 0x30, 0xd2, 0x82, 0x47, 0xc3, 0x02, 0xc3, 0x8b, 0x32, 0x43, 0x3b, 0xcf, 0x2d, 0xa7, 0x2b,
	0xb2, 0xc3, 0x12, 0x93, 0x26, 0x85, 0x06, 0x8f, 0x1b, 0xda, 0x83, 0xa2, 0x6e, 0xe4, 0xea, 0x67,
};

const uint8_t cjgdy_key[0x100] = {
	0x80, 0x88, 0x83, 0x9b, 0x87, 0x23, 0x87, 0x82, 0x1a, 0x9b, 0xb3, 0x12, 0x0c, 0x04, 0xa7, 0x26,
	0x11, 0x8b, 0x02, 0x13, 0x96, 0x11, 0x16, 0x06, 0x13, 0x9a, 0x1b, 0x8b, 0x8a, 0x9e, 0x9e, 0x82,
	0x22, 0xa3, 0xa2, 0x93, 0x25, 0xac, 0x02, 0x0a, 0x3a, 0x18, 0x3b, 0x2b, 0x0b, 0x29, 0x82, 0x22,
	0xb0, 0x29, 0x03, 0x8a, 0xb4, 0x15, 0x13, 0x0f, 0x08, 0x90, 0xa3, 0x92, 0x99, 0x29, 0xbb, 0x17,
	0xa2, 0x42, 0x22, 0xeb, 0xa4, 0x0f, 0x67, 0x4b, 0x10, 0xb3, 0x3a, 0x32, 0xab, 0x0b, 0xe7, 0x0a,
	0x93, 0x9a, 0x03, 0x4a, 0x82, 0x95, 0x07, 0xde, 0xd8, 0x42, 0x42, 0x83, 0x5b, 0x53, 0x8a, 0xcf,
	0x53, 0xdb, 0x93, 0xb3, 0xe6, 0xc9, 0x42, 0xe6, 0x8b, 0xe1, 0x0a, 0x12, 0xc1, 0x08, 0x62, 0x0b,
	0xd3, 0x72, 0x72, 0x8b, 0x77, 0x7e, 0x22, 0x43, 0x1a, 0x9a, 0xf2, 0x7a, 0x78, 0xc6, 0xaa, 0xf3,
	0x33, 0xa9, 0x23, 0x33, 0xa3, 0x00, 0x22, 0x0b, 0x11, 0x11, 0x83, 0x2b, 0xaf, 0x89, 0x26, 0x8e,
	0x93, 0x8a, 0x83, 0x8a, 0x02, 0x12, 0x96, 0x9e, 0x91, 0x80, 0x83, 0x9a, 0x1f, 0x07, 0x82, 0x0b,
	0xa0, 0xa1, 0x12, 0x1b, 0xa3, 0x2f, 0x03, 0xaa, 0x20, 0x28, 0x13, 0x33, 0x23, 0xad, 0xab, 0x27,
	0x03, 0x19, 0x33, 0x9b, 0x35, 0x19, 0xa6, 0x1f, 0x2a, 0xb9, 0x83, 0xb2, 0x92, 0x80, 0xa7, 0x1f,
	0xd2, 0xd9, 0x52, 0xfa, 0x47, 0x03, 0x07, 0x82, 0xf1, 0x79, 0x5a, 0x9b, 0x25, 0xed, 0xef, 0xae,
	0x13, 0x98, 0x52, 0x13, 0x90, 0x01, 0xc7, 0x86, 0x09, 0xd9, 0x4a, 0x92, 0x57, 0x19, 0x83, 0xd7,
	0x23, 0x71, 0xa3, 0x73, 0xc6, 0x26, 0x67, 0xa7, 0xc0, 0x79, 0xd3, 0x43, 0xa7, 0xe1, 0x63, 0x8f,
	0x60, 0xe2, 0x33, 0x32, 0x71, 0x8d, 0x82, 0x23, 0x72, 0x73, 0xd2, 0x83, 0x5f, 0x22, 0x5b, 0xdb,
};

const uint8_t xydn_key[0x100] = {
	0xb3, 0x23, 0x82, 0xb3, 0x86, 0x2a, 0x03, 0xaa, 0x0b, 0x08, 0x82, 0x83, 0x08, 0xab, 0x8e, 0x26,
	0x03, 0x92, 0x12, 0x1a, 0x12, 0x1f, 0x93, 0x8f, 0x11, 0x01, 0x03, 0x82, 0x04, 0x1f, 0x9a, 0x8f,
	0xb2, 0x12, 0xa3, 0xb3, 0x24, 0x21, 0xa3, 0x87, 0x18, 0x33, 0xab, 0x3a, 0xa8, 0xab, 0x06, 0x2f,
	0x93, 0x1b, 0x82, 0xa2, 0xa5, 0x30, 0xb2, 0x17, 0x99, 0x19, 0xab, 0x82, 0xb3, 0xb3, 0x37, 0x8a,
	0x33, 0xb8, 0xc3, 0xa2, 0x80, 0xe2, 0xc3, 0x26, 0xca, 0xb9, 0xb2, 0xb3, 0x6d, 0xeb, 0xea, 0xcf,
	0xd3, 0xc8, 0x92, 0xdb, 0x10, 0x93, 0xc7, 0x92, 0x0b, 0xc2, 0x93, 0x5a, 0xc1, 0x44, 0x5e, 0xc7,
	0xd0, 0x7a, 0xd2, 0x73, 0x05, 0x88, 0xc2, 0x0e, 0xb2, 0xf0, 0xba, 0x4b, 0xa6, 0x43, 0xa6, 0x2f,
	0xc2, 0xf2, 0x73, 0x22, 0x40, 0xd3, 0x72, 0xae, 0x6b, 0xc9, 0xe2, 0x5b, 0x15, 0x69, 0x0b, 0x8b,
	0x10, 0x89, 0x03, 0x12, 0x26, 0x29, 0xa6, 0xa7, 0x99, 0x38, 0x0b, 0x9b, 0x08, 0x8b, 0x8e, 0x8e,
	0x92, 0x91, 0x12, 0x13, 0x15, 0x89, 0x83, 0x0f, 0x12, 0x89, 0x13, 0x82, 0x08, 0x03, 0x8e, 0x9e,
	0x10, 0x09, 0xb2, 0xab, 0x81, 0x0a, 0x23, 0x0b, 0x01, 0x2a, 0x2a, 0x03, 0x2e, 0x84, 0x2e, 0x82,
	0x81, 0xb8, 0x02, 0x2b, 0x22, 0x95, 0x82, 0x2f, 0x2a, 0x9b, 0x02, 0x0b, 0x33, 0xb1, 0x07, 0x87,
	0xa0, 0x22, 0xb2, 0x63, 0x60, 0xc9, 0x62, 0xef, 0xe0, 0x51, 0x82, 0xd2, 0x6f, 0xe5, 0xea, 0xcf,
	0x40, 0x8b, 0x43, 0x92, 0x05, 0x9e, 0x47, 0xc7, 0x19, 0x9a, 0x4a, 0x83, 0x1c, 0x0b, 0xca, 0x13,
	0xf0, 0xfa, 0xc3, 0x5a, 0xc0, 0x2e, 0xc7, 0xcf, 0x71, 0x5a, 0xbb, 0xdb, 0xc2, 0x2e, 0xee, 0xa3,
	0x53, 0x38, 0x33, 0x7b, 0xf0, 0x20, 0x12, 0x8b, 0x92, 0x92, 0xb2, 0x0a, 0x5b, 0x4a, 0x2f, 0x87,
};

const uint8_t mjzhizun_key[0x100] = {
	0x23, 0x88, 0x02, 0x0b, 0x23, 0x0a, 0x87, 0xa3, 0xbb, 0x82, 0x0b, 0x0b, 0x83, 0x0d, 0xae, 0x87,
	0x93, 0x89, 0x93, 0x1a, 0x93, 0x82, 0x96, 0x06, 0x91, 0x81, 0x8b, 0x82, 0x13, 0x8c, 0x12, 0x82,
	0x12, 0x09, 0x03, 0x2b, 0x00, 0x0e, 0x87, 0x03, 0x88, 0x32, 0x3a, 0x9b, 0x89, 0x2c, 0xaf, 0x22,
	0x93, 0x88, 0x13, 0x23, 0x31, 0x02, 0x33, 0xab, 0x33, 0x39, 0x02, 0x1a, 0x0c, 0x9d, 0x0f, 0x1a,
	0x01, 0xd3, 0x93, 0xca, 0xe3, 0x6f, 0xe6, 0xe7, 0x2a, 0xb9, 0x93, 0xaa, 0xe0, 0xab, 0xc7, 0xa3,
	0x92, 0x5b, 0xc2, 0x12, 0x53, 0x87, 0x17, 0x9f, 0x48, 0xc0, 0xc2, 0x92, 0xcf, 0x85, 0x82, 0xd6,
	0xa3, 0xf1, 0x03, 0x5a, 0x82, 0xca, 0xe2, 0xee, 0x68, 0x19, 0x2a, 0xda, 0xad, 0x6a, 0x4a, 0xab,
	0xa0, 0xf9, 0x32, 0x13, 0xd0, 0x8a, 0xa6, 0x2a, 0x68, 0xd8, 0xcb, 0x42, 0x6d, 0x9c, 0xa3, 0xdb,
	0xb3, 0x32, 0x03, 0x13, 0x84, 0x06, 0x22, 0x83, 0x19, 0x1b, 0x8b, 0x1b, 0x21, 0xa8, 0x0f, 0x02,
	0x90, 0x9b, 0x83, 0x12, 0x95, 0x0d, 0x83, 0x96, 0x00, 0x9a, 0x93, 0x02, 0x1d, 0x1e, 0x8a, 0x93,
	0x01, 0x93, 0x23, 0xaa, 0xa6, 0xa8, 0x86, 0x26, 0x99, 0x33, 0x13, 0x23, 0xab, 0x2d, 0x2f, 0xae,
	0x13, 0x19, 0x93, 0x8a, 0x32, 0x07, 0x86, 0xaa, 0x13, 0xb9, 0x82, 0x2b, 0x07, 0x00, 0x8f, 0x86,
	0x02, 0x28, 0x82, 0x53, 0x83, 0x23, 0x86, 0x8b, 0xb9, 0xf8, 0x1b, 0x33, 0x04, 0x2d, 0x6a, 0x87,
	0x40, 0x0b, 0xc2, 0x53, 0x56, 0x5d, 0xc3, 0x8f, 0x5a, 0x0b, 0xc3, 0xd3, 0x44, 0x1d, 0x8a, 0x57,
	0x22, 0x50, 0xe2, 0xca, 0xc4, 0xa3, 0x42, 0xc2, 0xd1, 0xb3, 0x7b, 0x02, 0xeb, 0x61, 0x6b, 0x23,
	0x20, 0xe8, 0xd3, 0xc2, 0xc1, 0x65, 0xa2, 0xeb, 0x00, 0x6a, 0xe2, 0x8b, 0xb3, 0xc6, 0x13, 0xb6,
};

const uint8_t lhfy_key[0x100] = {
	0x23, 0x1b, 0x32, 0x9a, 0x25, 0x85, 0x86, 0x03, 0xb0, 0x30, 0x1a, 0x1b, 0x0c, 0xa2, 0x2b, 0x8b,
	0x81, 0x9b, 0x83, 0x13, 0x17, 0x8d, 0x16, 0x9a, 0x0b, 0x1a, 0x9a, 0x1b, 0x9a, 0x8b, 0x1f, 0x8b,
	0x31, 0x03, 0x12, 0x23, 0x20, 0x89, 0x87, 0x8b, 0x21, 0x99, 0xa2, 0x2b, 0xac, 0x2c, 0xa7, 0xa7,
	0x91, 0x88, 0x12, 0x33, 0x21, 0x2d, 0xa7, 0x9a, 0x98, 0xb9, 0xbb, 0x8b, 0x2f, 0x1e, 0x8a, 0x07,
	0x00, 0x29, 0xc2, 0x83, 0xa2, 0x48, 0xc3, 0x26, 0xb3, 0xca, 0x5b, 0x93, 0x03, 0xe3, 0x2f, 0x62,
	0x02, 0x11, 0x52, 0x8a, 0x85, 0x97, 0xd6, 0xdf, 0x19, 0x5b, 0x12, 0xdb, 0x58, 0x17, 0xde, 0x8b,
	0x91, 0x88, 0xe2, 0x63, 0x42, 0xad, 0xc2, 0x4a, 0xe8, 0x22, 0xd2, 0x0b, 0x29, 0x2b, 0x43, 0xab,
	0xb1, 0x9a, 0x53, 0x42, 0x03, 0x39, 0x42, 0xaa, 0xe1, 0xda, 0xc2, 0xaa, 0xfe, 0x14, 0x1e, 0x27,
	0xa3, 0x3a, 0x23, 0x32, 0xa0, 0xaf, 0x87, 0x2a, 0xa8, 0x03, 0xb2, 0x02, 0xa0, 0x8c, 0x27, 0x8f,
	0x00, 0x98, 0x93, 0x9a, 0x03, 0x94, 0x92, 0x0a, 0x18, 0x13, 0x92, 0x8b, 0x09, 0x0e, 0x1b, 0x1a,
	0x82, 0x91, 0x22, 0x23, 0xa5, 0x8e, 0xa3, 0xae, 0xb0, 0x12, 0x82, 0x9a, 0xa0, 0x8e, 0x27, 0x2f,
	0x31, 0x91, 0x23, 0xa2, 0xb6, 0x20, 0x32, 0x82, 0x1b, 0x33, 0x33, 0x33, 0x9f, 0x3d, 0x83, 0xb2,
	0x03, 0xf3, 0xa3, 0x4b, 0xe0, 0xa9, 0xa2, 0x42, 0xd8, 0xfb, 0x5a, 0x0a, 0x60, 0x88, 0x07, 0xea,
	0x90, 0x93, 0x83, 0x1a, 0xc3, 0x16, 0x42, 0xda, 0xc0, 0xc0, 0x9b, 0x5a, 0x8a, 0x91, 0x4e, 0xd6,
	0x51, 0x62, 0x63, 0xe3, 0x61, 0xea, 0xc2, 0x4f, 0x3a, 0x6b, 0x02, 0xc3, 0xa8, 0xec, 0xa2, 0xa3,
	0xd3, 0xf9, 0x82, 0xd3, 0x62, 0xa4, 0xb3, 0xdf, 0xc2, 0xcb, 0x0a, 0x9b, 0x09, 0x92, 0xde, 0x6e,
};

const uint8_t lhzbgqb_key[0x100] = {
	0xa3, 0x22, 0x03, 0x03, 0x02, 0xa7, 0xa6, 0x8f, 0x92, 0x92, 0x8a, 0x1b, 0xac, 0x23, 0x0f, 0xaf,
	0x10, 0x1a, 0x12, 0x0a, 0x82, 0x16, 0x92, 0x8b, 0x1a, 0x00, 0x0b, 0x9a, 0x9f, 0x86, 0x8b, 0x0f,
	0x20, 0xa3, 0x32, 0x2b, 0x00, 0xa6, 0x02, 0x23, 0x2a, 0x88, 0x3a, 0xab, 0x23, 0x2b, 0x26, 0x0b,
	0x82, 0x2b, 0x12, 0xab, 0x15, 0x25, 0x82, 0xa2, 0x92, 0xb9, 0x93, 0x32, 0x15, 0x94, 0x2e, 0x1e,
	0x60, 0x12, 0x72, 0x02, 0x64, 0x25, 0xa2, 0xeb, 0xa2, 0x43, 0x6b, 0x0a, 0x6d, 0x4a, 0x62, 0x07,
	0x82, 0x1b, 0x12, 0x83, 0xc5, 0x4a, 0x83, 0x0b, 0xcb, 0x4b, 0xc3, 0x92, 0x95, 0x09, 0x4b, 0xdb,
	0x32, 0xa0, 0x32, 0x1b, 0xe5, 0xc6, 0x27, 0x22, 0xda, 0x0a, 0x1b, 0x9b, 0xcc, 0x2c, 0xa2, 0x66,
	0x91, 0x52, 0xb3, 0x4a, 0x25, 0x1d, 0x26, 0x86, 0x49, 0x23, 0xc2, 0xf3, 0x16, 0x8f, 0xde, 0xe7,
	0x33, 0x2a, 0x32, 0x9b, 0xa7, 0xa1, 0x22, 0xa3, 0x83, 0x8b, 0xba, 0xa3, 0xa9, 0xa7, 0x27, 0x87,
	0x01, 0x88, 0x03, 0x0a, 0x92, 0x96, 0x03, 0x02, 0x9a, 0x8b, 0x0b, 0x8a, 0x88, 0x01, 0x1f, 0x83,
	0xa2, 0x9a, 0x33, 0x92, 0xa5, 0xae, 0x23, 0xa3, 0x00, 0x31, 0x02, 0x0a, 0x0b, 0xa6, 0x8f, 0x8f,
	0x20, 0x18, 0x83, 0x3a, 0x83, 0x3c, 0x82, 0x3e, 0x20, 0x08, 0x9b, 0x1a, 0xb3, 0xb7, 0x83, 0xbf,
	0x13, 0x21, 0xe3, 0x5a, 0x44, 0xc5, 0xc2, 0xcf, 0xd9, 0x20, 0x2a, 0x4b, 0x02, 0x81, 0xaa, 0x87,
	0x82, 0x08, 0x03, 0x92, 0x11, 0x59, 0x03, 0xcf, 0x12, 0x4a, 0x0a, 0x8a, 0x98, 0x89, 0x83, 0x86,
	0xc2, 0x63, 0x42, 0x4a, 0x03, 0x2d, 0xe6, 0x2b, 0x5a, 0x8b, 0xea, 0x02, 0xce, 0xa8, 0x6f, 0xa6,
	0xe1, 0x2a, 0xc3, 0xfa, 0x94, 0x44, 0xb3, 0xce, 0x81, 0xe9, 0xf3, 0x5b, 0x46, 0x0c, 0x16, 0xa2,
};

const uint8_t mghammer_key[0x100] = {
	0x92, 0x8b, 0x33, 0x1b, 0x06, 0x25, 0x02, 0xaa, 0x1b, 0x89, 0xb3, 0x8b, 0x8e, 0xaf, 0xae, 0x2e,
	0x91, 0x11, 0x92, 0x83, 0x01, 0x14, 0x83, 0x87, 0x88, 0x00, 0x93, 0x03, 0x8c, 0x82, 0x02, 0x93,
	0xa2, 0x83, 0x82, 0xb3, 0x25, 0x8c, 0x86, 0x87, 0x3a, 0x98, 0xa2, 0x9b, 0x08, 0x08, 0x2b, 0xa3,
	0x10, 0x9b, 0xa2, 0x1a, 0x31, 0x30, 0xa2, 0x9f, 0xb8, 0x20, 0x3b, 0x02, 0x9a, 0x81, 0x3b, 0xae,
	0xd2, 0x1b, 0xe3, 0x22, 0x61, 0x2d, 0x07, 0xea, 0xa2, 0x3b, 0x73, 0x7b, 0x23, 0xa9, 0xca, 0x4a,
	0x53, 0xd0, 0x53, 0x42, 0x07, 0x8f, 0x46, 0x4b, 0x9a, 0x82, 0xc2, 0x12, 0x5f, 0x45, 0xcf, 0x9a,
	0xa2, 0x33, 0x52, 0x5b, 0x26, 0xc5, 0x86, 0xa7, 0x13, 0x30, 0xab, 0xd2, 0x0d, 0x44, 0x2e, 0x63,
	0x32, 0xb1, 0x03, 0xc2, 0x92, 0xbb, 0xf6, 0x93, 0x79, 0x48, 0x9b, 0x4a, 0x4c, 0xe0, 0x13, 0xae,
	0xa1, 0x22, 0xb2, 0x12, 0x81, 0xae, 0x87, 0xa7, 0x82, 0x32, 0x12, 0xbb, 0x8e, 0xa8, 0x82, 0x2e,
	0x00, 0x10, 0x93, 0x03, 0x86, 0x1a, 0x02, 0x83, 0x1a, 0x1b, 0x8b, 0x1a, 0x80, 0x11, 0x83, 0x86,
	0xa0, 0x98, 0x23, 0xba, 0x85, 0x8e, 0x07, 0x8b, 0x33, 0x39, 0x32, 0x93, 0x83, 0xaf, 0x0e, 0x07,
	0x92, 0x2a, 0x22, 0x2a, 0x80, 0x96, 0x03, 0xba, 0x31, 0x08, 0x13, 0x8b, 0x93, 0x88, 0x97, 0x17,
	0x23, 0x19, 0xd2, 0x32, 0x22, 0xef, 0x02, 0xc2, 0x12, 0x71, 0x1a, 0x2a, 0x6e, 0xa0, 0x47, 0x46,
	0x93, 0xd0, 0xd2, 0x43, 0x03, 0x96, 0x17, 0xcb, 0xd9, 0x51, 0x42, 0xc2, 0x16, 0x45, 0xc3, 0xdb,
	0x42, 0x51, 0xe3, 0x2b, 0x44, 0x08, 0x42, 0xa2, 0x9a, 0xf2, 0xc2, 0x5a, 0x4a, 0x2b, 0x03, 0x82,
	0x31, 0xc8, 0x13, 0x93, 0xe3, 0x01, 0xe6, 0xd3, 0xa3, 0xa0, 0x1b, 0x02, 0x6b, 0x98, 0x02, 0xc7,
};

const uint8_t mghammer_io_key[0x100] = {
	0x92, 0x8b, 0x33, 0x1b, 0x06, 0x25, 0x02, 0xaa, 0x1b, 0x89, 0xb3, 0x8b, 0x8e, 0xaf, 0xae, 0x2e,
	0x91, 0x11, 0x92, 0x83, 0x01, 0x14, 0x83, 0x87, 0x88, 0x00, 0x93, 0x03, 0x8c, 0x82, 0x02, 0x93,
	0xa2, 0x83, 0x82, 0xb3, 0x25, 0x8c, 0x86, 0x87, 0x3a, 0x98, 0xa2, 0x9b, 0x08, 0x08, 0x2b, 0xa3,
	0x10, 0x9b, 0xa2, 0x1a, 0x31, 0x30, 0xa2, 0x9f, 0xb8, 0x20, 0x3b, 0x02, 0x9a, 0x81, 0x3b, 0xae,
	0xd2, 0x1b, 0xe3, 0x22, 0x61, 0x2d, 0x07, 0xea, 0xa2, 0x3b, 0x73, 0x7b, 0x23, 0xa9, 0xca, 0x4a,
	0x53, 0xd0, 0x53, 0x42, 0x07, 0x8f, 0x46, 0x4b, 0x9a, 0x82, 0xc2, 0x12, 0x5f, 0x45, 0xcf, 0x9a,
	0xa2, 0x33, 0x52, 0x5b, 0x26, 0xc5, 0x86, 0xa7, 0x13, 0x30, 0xab, 0xd2, 0x0d, 0x44, 0x2e, 0x63,
	0x32, 0xb1, 0x03, 0xc2, 0x92, 0xbb, 0xf6, 0x93, 0x79, 0x48, 0x9b, 0x4a, 0x4c, 0xe0, 0x13, 0xae,
	0xa1, 0x22, 0xb2, 0x12, 0x81, 0xae, 0x87, 0xa7, 0x82, 0x32, 0x12, 0xbb, 0x8e, 0xa8, 0x82, 0x2e,
	0x00, 0x10, 0x93, 0x03, 0x86, 0x1a, 0x02, 0x83, 0x1a, 0x1b, 0x8b, 0x1a, 0x80, 0x11, 0x83, 0x86,
	0xa0, 0x98, 0x23, 0xba, 0x85, 0x8e, 0x07, 0x8b, 0x33, 0x39, 0x32, 0x93, 0x83, 0xaf, 0x0e, 0x07,
	0x92, 0x2a, 0x22, 0x2a, 0x80, 0x96, 0x03, 0xba, 0x31, 0x08, 0x13, 0x8b, 0x93, 0x88, 0x97, 0x17,
	0x23, 0x19, 0xd2, 0x32, 0x22, 0xef, 0x02, 0xc2, 0x12, 0x71, 0x1a, 0x2a, 0x6e, 0xa0, 0x47, 0x46,
	0x93, 0xd0, 0xd2, 0x43, 0x03, 0x96, 0x17, 0xcb, 0xd9, 0x51, 0x42, 0xc2, 0x16, 0x45, 0xc3, 0xdb,
	0x42, 0x51, 0xe3, 0x2b, 0x44, 0x08, 0x42, 0xa2, 0x9a, 0xf2, 0xc2, 0x5a, 0x4a, 0x2b, 0x03, 0x82,
	0x31, 0xc8, 0x13, 0x93, 0xe3, 0x01, 0xe6, 0xd3, 0xa3, 0xa0, 0x1b, 0x02, 0x6b, 0x98, 0x02, 0xc7,
};

const uint8_t zjddz_key[0x100] = {
	0x01, 0x2a, 0x02, 0x1a, 0xa7, 0x82, 0xa2, 0x8b, 0xa1, 0x80, 0x3b, 0x8a, 0xa5, 0xa8, 0x83, 0x0b,
	0x11, 0x18, 0x83, 0x92, 0x86, 0x15, 0x97, 0x97, 0x00, 0x11, 0x1b, 0x03, 0x8c, 0x86, 0x1e, 0x9e,
	0x11, 0x2b, 0x82, 0xa2, 0x21, 0x22, 0xa2, 0xa3, 0xb0, 0x83, 0x3a, 0xa2, 0x8f, 0xad, 0x02, 0xa7,
	0xb0, 0x38, 0x22, 0x0b, 0x13, 0xa7, 0xa3, 0x03, 0x38, 0x00, 0x23, 0x0a, 0x02, 0x99, 0x86, 0x12,
	0x83, 0x10, 0x72, 0x93, 0xe2, 0x67, 0x43, 0xce, 0x0a, 0xd3, 0xab, 0x4a, 0x20, 0xc9, 0x83, 0x62,
	0x51, 0x89, 0x83, 0xc2, 0xd1, 0x40, 0x47, 0xce, 0xda, 0xd2, 0x9b, 0x9b, 0xcf, 0x5a, 0x96, 0x4f,
	0xf2, 0xf2, 0xa3, 0x9a, 0xe4, 0xec, 0x43, 0xe2, 0xe1, 0x91, 0x7a, 0x3b, 0xac, 0x8e, 0xae, 0xef,
	0x40, 0x73, 0xc2, 0xb3, 0xd1, 0x7a, 0xc2, 0x0b, 0x53, 0x9a, 0xf3, 0x13, 0x5c, 0x61, 0xb7, 0x37,
	0x82, 0x19, 0xb2, 0x9b, 0x27, 0x01, 0x06, 0x22, 0x18, 0x82, 0xb3, 0x2b, 0x0b, 0x24, 0x2e, 0x86,
	0x10, 0x01, 0x03, 0x0a, 0x92, 0x0c, 0x12, 0x97, 0x19, 0x01, 0x0b, 0x0a, 0x18, 0x86, 0x8f, 0x02,
	0x90, 0x29, 0xb3, 0xa3, 0x04, 0x8e, 0x83, 0x0f, 0x23, 0x11, 0x22, 0x8a, 0xae, 0x05, 0x8a, 0x8b,
	0x02, 0x90, 0x03, 0x32, 0xb1, 0x8f, 0xa7, 0x8f, 0x80, 0xa3, 0x13, 0x12, 0x18, 0x20, 0x8a, 0x3e,
	0x00, 0x31, 0xb3, 0xaa, 0x03, 0x20, 0x22, 0x27, 0xdb, 0x4b, 0x7b, 0x63, 0xad, 0x07, 0x2e, 0xaa,
	0x02, 0xd8, 0x52, 0x52, 0x11, 0x5e, 0x17, 0xca, 0x13, 0x08, 0x4b, 0x82, 0x47, 0x48, 0x0e, 0xca,
	0xc1, 0x1a, 0xe2, 0xf2, 0xc6, 0x8e, 0xc3, 0x62, 0x2b, 0x88, 0xa2, 0xa2, 0x0b, 0xe2, 0xae, 0x46,
	0xb2, 0x90, 0xe2, 0x82, 0x82, 0x41, 0xd2, 0xdf, 0xd2, 0x2b, 0xda, 0xbb, 0xf6, 0xd5, 0xc3, 0xb7,
};

const uint8_t hyleyuan_key[0x100] = {
	0xa0, 0x02, 0xb0, 0x21, 0x26, 0x2e, 0x85, 0xa5, 0x90, 0xb9, 0x21, 0x81, 0x04, 0x80, 0x89, 0x89,
	0x12, 0x83, 0x81, 0x98, 0x12, 0x01, 0x10, 0x89, 0x01, 0x82, 0x08, 0x90, 0x03, 0x94, 0x01, 0x89,
	0x12, 0xb9, 0x11, 0x81, 0x02, 0x86, 0x00, 0x09, 0x80, 0x9a, 0x11, 0x91, 0x27, 0x07, 0x2d, 0xa8,
	0x80, 0x29, 0x80, 0x10, 0x95, 0x9c, 0x24, 0x90, 0xa8, 0x81, 0x31, 0xb9, 0x17, 0x2f, 0x9c, 0x35,
	0x72, 0x7b, 0x30, 0x88, 0x04, 0xeb, 0xc4, 0x08, 0x73, 0xb2, 0xe1, 0x90, 0x04, 0xc9, 0x80, 0x2d,
	0xc2, 0xc9, 0x81, 0xc0, 0x85, 0x9a, 0x41, 0x8d, 0x11, 0x02, 0x90, 0x98, 0xd7, 0x0a, 0x84, 0x49,
	0xa1, 0x79, 0x51, 0xd9, 0x23, 0x6a, 0x80, 0xa9, 0x68, 0x50, 0x08, 0x70, 0x40, 0x06, 0x2c, 0xa8,
	0xb1, 0xf9, 0xe0, 0x11, 0x25, 0x14, 0xe4, 0x28, 0xe3, 0x82, 0x88, 0x99, 0x29, 0x04, 0x1c, 0x65,
	0xa0, 0x2a, 0xb0, 0x18, 0x03, 0x82, 0x85, 0x04, 0x0b, 0x80, 0x91, 0x80, 0xa2, 0x88, 0x01, 0x81,
	0x83, 0x91, 0x91, 0x01, 0x13, 0x81, 0x95, 0x88, 0x88, 0x19, 0x98, 0x81, 0x8e, 0x0e, 0x9d, 0x19,
	0x90, 0x2a, 0x90, 0x90, 0x24, 0x88, 0x85, 0x20, 0x03, 0x3b, 0x38, 0x20, 0x04, 0x04, 0xa1, 0x84,
	0x92, 0x08, 0x01, 0x19, 0x31, 0x16, 0x84, 0x29, 0x22, 0xbb, 0x99, 0x80, 0x15, 0x37, 0xbd, 0x85,
	0xe0, 0x2a, 0xb1, 0x49, 0x22, 0xc4, 0x61, 0xc5, 0x3a, 0xc1, 0xb0, 0x28, 0x2c, 0x0c, 0x85, 0xc9,
	0x41, 0xdb, 0xc1, 0xd1, 0x14, 0x4e, 0x05, 0xdc, 0x03, 0x02, 0x48, 0x90, 0x48, 0xc0, 0x08, 0x18,
	0x30, 0x51, 0xb0, 0xd1, 0x24, 0xa1, 0x20, 0x40, 0x32, 0xb9, 0xf8, 0x68, 0xa3, 0xc8, 0x88, 0x05,
	0x80, 0xe1, 0x11, 0xe1, 0x40, 0x63, 0x84, 0x90, 0x11, 0x49, 0xd1, 0x29, 0x45, 0x01, 0x35, 0x08,
};

const uint8_t hlddz_key[0x100] = {
	0xb1, 0x02, 0x20, 0x29, 0x05, 0xa7, 0xa5, 0x2c, 0xbb, 0x9a, 0x09, 0x20, 0x09, 0x87, 0x85, 0x28,
	0x13, 0x80, 0x11, 0x10, 0x03, 0x1a, 0x04, 0x81, 0x0b, 0x0a, 0x10, 0x08, 0x14, 0x01, 0x91, 0x04,
	0x80, 0x18, 0x31, 0x38, 0x20, 0x05, 0xa5, 0xac, 0x82, 0xb1, 0x21, 0x91, 0x2a, 0x2a, 0x01, 0x0d,
	0xa1, 0xb2, 0x10, 0x31, 0x21, 0x33, 0x20, 0x28, 0x12, 0xba, 0x38, 0xb1, 0x81, 0x8d, 0x01, 0x94,
	0xe1, 0x48, 0x10, 0xd8, 0x81, 0x83, 0x44, 0x6c, 0xba, 0x02, 0x80, 0x29, 0x68, 0x23, 0xa1, 0x41,
	0x43, 0xcb, 0x50, 0xd0, 0xc3, 0x06, 0x85, 0xd1, 0x91, 0x8a, 0xd9, 0x10, 0x08, 0xce, 0xd4, 0x58,
	0xb0, 0x21, 0x91, 0xd1, 0x22, 0x89, 0x81, 0x68, 0x3a, 0x31, 0x89, 0xf9, 0x66, 0x25, 0x21, 0x28,
	0x42, 0x00, 0xf1, 0x71, 0x32, 0xcb, 0x25, 0xcc, 0x90, 0xb1, 0xa9, 0x28, 0xf9, 0xf3, 0xb8, 0x79,
	0x12, 0x98, 0x81, 0x18, 0xa4, 0x2d, 0x84, 0x08, 0xa1, 0x99, 0x10, 0x09, 0x21, 0x2b, 0xad, 0xa0,
	0x11, 0x00, 0x80, 0x98, 0x15, 0x9c, 0x01, 0x1d, 0x19, 0x11, 0x08, 0x19, 0x15, 0x9d, 0x94, 0x18,
	0xa3, 0x08, 0x91, 0x30, 0x86, 0x28, 0x01, 0xa4, 0x08, 0x28, 0xb1, 0x19, 0x0b, 0x8b, 0x8d, 0x8c,
	0x82, 0x39, 0xb1, 0x31, 0x34, 0x3f, 0x11, 0x35, 0x8b, 0x33, 0x08, 0x18, 0x87, 0x37, 0x9c, 0x20,
	0xe2, 0xab, 0x41, 0x91, 0xc1, 0xe1, 0x60, 0xe5, 0xf9, 0xa3, 0xf9, 0x98, 0x2c, 0x0e, 0x05, 0x05,
	0x11, 0x51, 0xd1, 0xc9, 0x40, 0x4d, 0xd5, 0x8c, 0xdb, 0x1b, 0x48, 0x89, 0x84, 0xc7, 0x58, 0x9c,
	0x00, 0x52, 0x30, 0x71, 0x04, 0x21, 0x05, 0x00, 0x9b, 0x79, 0x88, 0xf0, 0xe3, 0x2f, 0xcc, 0x05,
	0xf3, 0xe8, 0x41, 0x79, 0x90, 0x7d, 0xd5, 0x1c, 0x83, 0x20, 0xd8, 0x38, 0xeb, 0x97, 0x65, 0x11,
};

const uint8_t cjjl_key[0x100] = {
	0xb2, 0x3b, 0x92, 0x12, 0x05, 0x20, 0x23, 0x8b, 0x29, 0xa1, 0xba, 0x82, 0x8f, 0x89, 0x8a, 0x0f,
	0x93, 0x81, 0x82, 0x9a, 0x00, 0x0e, 0x13, 0x0b, 0x09, 0x01, 0x1b, 0x02, 0x86, 0x80, 0x13, 0x0b,
	0x00, 0x2b, 0x92, 0x82, 0x80, 0x0a, 0xa6, 0x27, 0xa0, 0x2a, 0xba, 0x03, 0x81, 0xa2, 0x83, 0x8a,
	0x10, 0x22, 0x83, 0x2a, 0xb0, 0x8d, 0xa3, 0x27, 0x00, 0xa8, 0x1b, 0x23, 0x17, 0x25, 0x87, 0xa6,
	0x12, 0x93, 0xc2, 0xeb, 0xc3, 0xc9, 0xc7, 0x42, 0x3b, 0x21, 0xe2, 0x73, 0xc5, 0x86, 0xa2, 0x87,
	0x82, 0x88, 0x93, 0xda, 0x43, 0x4e, 0x53, 0x47, 0x51, 0xca, 0x4b, 0x4b, 0x47, 0xde, 0xde, 0x8f,
	0x71, 0xf2, 0x02, 0xba, 0x05, 0xc6, 0x02, 0xea, 0x79, 0x10, 0xca, 0x02, 0xa7, 0x82, 0x0b, 0x86,
	0x63, 0x28, 0x22, 0x9a, 0x32, 0x12, 0x77, 0xa7, 0xb9, 0xf0, 0x42, 0xf3, 0xb1, 0x22, 0x32, 0xa3,
	0x00, 0x11, 0x22, 0x83, 0xa7, 0x02, 0x03, 0x83, 0x98, 0xaa, 0x22, 0xb2, 0x8c, 0x0e, 0x82, 0x0f,
	0x92, 0x09, 0x13, 0x92, 0x05, 0x10, 0x06, 0x07, 0x13, 0x01, 0x82, 0x1b, 0x0a, 0x9b, 0x16, 0x8b,
	0x03, 0xaa, 0xa2, 0xba, 0x20, 0x24, 0x83, 0x0e, 0xbb, 0x38, 0x8b, 0x3b, 0x83, 0x81, 0xae, 0x0e,
	0xa0, 0x90, 0x12, 0x23, 0x33, 0x28, 0x32, 0xb6, 0xab, 0x29, 0x3a, 0xaa, 0xae, 0x3f, 0x26, 0x37,
	0xf3, 0xe0, 0x62, 0x72, 0xa1, 0xac, 0x22, 0x43, 0xc8, 0x98, 0xbb, 0x4b, 0xcd, 0x8b, 0x0b, 0xa3,
	0x13, 0xc0, 0x53, 0x53, 0x44, 0xc2, 0x13, 0x0a, 0x0a, 0x08, 0x02, 0x4a, 0xd4, 0x16, 0xdb, 0x97,
	0x10, 0x00, 0xc2, 0xb2, 0x41, 0x63, 0x27, 0x8a, 0xab, 0x39, 0x23, 0xaa, 0xc5, 0xa6, 0x4f, 0x22,
	0x73, 0x11, 0xe2, 0xf2, 0xd1, 0x6f, 0xc7, 0xaa, 0xd8, 0xeb, 0x5b, 0x13, 0xf5, 0x9a, 0x3a, 0x62,
};
