!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2025 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief collects all constants needed in input so that they can be used without
!>      circular dependencies
!> \par History
!>      02.2005 created [fawzi&Joost]
!>      10.2008 Teodoro Laino [tlaino] - University of Zurich
!>              Cleaned this file of all arrays of strings. Here must be kept
!>              only integer/real global constants
! **************************************************************************************************
MODULE input_constants

   USE kinds,                           ONLY: dp

   IMPLICIT NONE

   PRIVATE

   ! Tags for development/release versions
   INTEGER, PARAMETER, PUBLIC               :: id_development_version = 0, &
                                               id_release_version = 1

   ! Program Names
   INTEGER, PARAMETER, PUBLIC               :: do_atom = 1, &
                                               do_farming = 2, &
                                               do_test = 3, &
                                               do_cp2k = 4, &
                                               do_optimize_input = 5, &
                                               do_opt_basis = 6, &
                                               do_tree_mc = 7, &
                                               do_tree_mc_ana = 8, &
                                               do_swarm = 9

   INTEGER, PARAMETER, PUBLIC               :: opt_force_matching = 1

   INTEGER, PARAMETER, PUBLIC               :: do_full_density = 0, &
                                               do_spin_density = 1

   INTEGER, PARAMETER, PUBLIC               :: do_no_et = 0, &
                                               do_et_ddapc = 1
   ! Method Names
   INTEGER, PARAMETER, PUBLIC               :: do_qs = 1, &
                                               do_fist = 2, &
                                               do_qmmm = 3, &
                                               do_qmmmx = 4, &
                                               do_eip = 5, &
                                               do_mixed = 6, &
                                               do_embed = 7, &
                                               do_sirius = 8, &
                                               do_nnp = 9, &
                                               do_ipi = 10

   ! QMMM
   REAL(KIND=dp), PARAMETER, PUBLIC         :: RADIUS_QMMM_DEFAULT = 0.80_dp, & ! Angstrom
                                               ALPHA_IMOMM_DEFAULT = 1.38_dp, &
                                               CHARGE_SCALE_FACTOR = 1.00_dp
   ! BAND
   INTEGER, PARAMETER, PUBLIC               :: band_md_opt = 0, &
                                               band_diis_opt = 1

   INTEGER, PARAMETER, PUBLIC               :: do_band_cartesian = 0, &
                                               do_band_collective = 1
   ! Weight_types
   INTEGER, PARAMETER, PUBLIC               :: weight_type_unit = 0, &
                                               weight_type_mass = 1

   ! Diagonalization routines of the ScaLAPACK library
   INTEGER, PARAMETER, PUBLIC               :: do_diag_syevd = 1, &
                                               do_diag_syevx = 2
   ! initial matrix methods methods
   INTEGER, PARAMETER, PUBLIC               :: do_mat_random = 1, &
                                               do_mat_read = 2

   INTEGER, PARAMETER, PUBLIC               :: do_constr_none = 1, &
                                               do_constr_atomic = 2, &
                                               do_constr_molec = 3

   INTEGER, PARAMETER, PUBLIC               :: do_cell_cp2k = 1, &
                                               do_cell_cif = 2, &
                                               do_cell_xsc = 3

   INTEGER, PARAMETER, PUBLIC               :: rmsd_all = 0, &
                                               rmsd_list = 1, &
                                               rmsd_weightlist = 2

   INTEGER, PARAMETER, PUBLIC               :: do_rep_interleaved = 0, &
                                               do_rep_blocked = 1

   ! MD Ensemble
   INTEGER, PARAMETER, PUBLIC               :: nve_ensemble = 1, &
                                               nvt_ensemble = 2, &
                                               npt_i_ensemble = 3, &
                                               npt_f_ensemble = 4, &
                                               nph_uniaxial_ensemble = 5, &
                                               nph_uniaxial_damped_ensemble = 6, &
                                               nph_ensemble = 7, &
                                               isokin_ensemble = 8, &
                                               reftraj_ensemble = 9, &
                                               langevin_ensemble = 10, &
                                               npe_f_ensemble = 11, &
                                               npe_i_ensemble = 12, &
                                               nvt_adiabatic_ensemble = 13, &
                                               npt_ia_ensemble = 14

   INTEGER, PARAMETER, PUBLIC               :: do_thermo_same_as_part = 100, &
                                               do_thermo_nose = 101, &
                                               do_thermo_csvr = 102, &
                                               do_thermo_gle = 103, &
                                               do_thermo_al = 104

   ! MD initialisation method
   INTEGER, PARAMETER, PUBLIC               :: md_init_default = 1, &
                                               md_init_vib = 2

   ! Run Types
   INTEGER, PARAMETER, PUBLIC               :: none_run = 1, &
                                               energy_run = 2, &
                                               energy_force_run = 3, &
                                               mol_dyn_run = 4, &
                                               geo_opt_run = 5, &
                                               mon_car_run = 6, &
                                               electronic_spectra_run = 7, &
                                               debug_run = 8, &
                                               bsse_run = 9, &
                                               linear_response_run = 10, &
                                               pint_run = 11, &
                                               vib_anal = 12, &
                                               do_band = 13, &
                                               cell_opt_run = 14, &
                                               real_time_propagation = 15, &
                                               ehrenfest = 16, &
                                               do_tamc = 17, &
                                               tree_mc_run = 18, &
                                               driver_run = 19, &
                                               negf_run = 20

   ! Run Types of Atom Code
   INTEGER, PARAMETER, PUBLIC               :: atom_no_run = 1, &
                                               atom_energy_run = 2, &
                                               atom_basis_run = 3, &
                                               atom_pseudo_run = 4

   ! Integral types in Atom Code
   INTEGER, PARAMETER, PUBLIC               :: do_analytic = 1, &
                                               do_semi_analytic = 2, &
                                               do_numeric = 3
   ! Basis set type for Atom Code
   INTEGER, PARAMETER, PUBLIC               :: gaussian = 1, &
                                               geometrical_gto = 2, &
                                               contracted_gto = 3, &
                                               slater = 4, &
                                               numerical = 5
   ! Method type for Atom Code
   INTEGER, PARAMETER, PUBLIC               :: do_rks_atom = 1, &
                                               do_uks_atom = 2, &
                                               do_rhf_atom = 3, &
                                               do_uhf_atom = 4, &
                                               do_rohf_atom = 5
   ! Relativistic correction for Atom Code
   INTEGER, PARAMETER, PUBLIC               :: do_nonrel_atom = 100, &
                                               do_zoramp_atom = 101, &
                                               do_sczoramp_atom = 102, &
                                               do_dkh0_atom = 110, &
                                               do_dkh1_atom = 111, &
                                               do_dkh2_atom = 112, &
                                               do_dkh3_atom = 113

   ! Pseudopotential type for Atom Code
   INTEGER, PARAMETER, PUBLIC               :: no_pseudo = 0, &
                                               gth_pseudo = 1, &
                                               upf_pseudo = 2, &
                                               ecp_pseudo = 3, &
                                               sgp_pseudo = 4
   ! Confinement type for Atom Code
   INTEGER, PARAMETER, PUBLIC               :: no_conf = 0, &
                                               poly_conf = 1, &
                                               barrier_conf = 2
   ! BAND
   INTEGER, PARAMETER, PUBLIC               :: do_b_neb = 0, &
                                               do_it_neb = 1, &
                                               do_ci_neb = 2, &
                                               do_d_neb = 3, &
                                               do_sm = 4, &
                                               do_eb = 5

   INTEGER, PARAMETER, PUBLIC               :: pot_neb_full = 0, &
                                               pot_neb_fe = 1, &
                                               pot_neb_me = 2

   INTEGER, PARAMETER, PUBLIC               :: cholesky_off = 0, &
                                               cholesky_reduce = 1, &
                                               cholesky_restore = 2, &
                                               cholesky_inverse = 3, &
                                               cholesky_dbcsr = 4

   INTEGER, PARAMETER, PUBLIC               :: atomic_guess = 1, &
                                               restart_guess = 2, &
                                               random_guess = 3, &
                                               core_guess = 4, &
                                               history_guess = 5, &
                                               no_guess = 6, &
                                               mopac_guess = 7, &
                                               sparse_guess = 8, &
                                               almo_scf_guess = 9, &
                                               molecular_guess = 10, &
                                               eht_guess = 11

   ! Projection formulas for the maximum overlap method
   INTEGER, PARAMETER, PUBLIC               :: momproj_norm = 0, &
                                               momproj_sum = 1

   ! Type of the MOM method
   INTEGER, PARAMETER, PUBLIC               :: momtype_mom = 0, &
                                               momtype_imom = 1

   INTEGER, PARAMETER, PUBLIC               :: almo_frz_none = 0, &
                                               almo_frz_isolated = 1, &
                                               almo_frz_crystal = 2, &
                                               almo_frz_both = 3

   INTEGER, PARAMETER, PUBLIC               :: do_gapw_gcs = 1, &
                                               do_gapw_gct = 2, &
                                               do_gapw_log = 3

   INTEGER, PARAMETER, PUBLIC               :: gapw_1c_orb = 1000, &
                                               gapw_1c_small = 1002, &
                                               gapw_1c_medium = 1003, &
                                               gapw_1c_large = 1004, &
                                               gapw_1c_very_large = 1005

   INTEGER, PARAMETER, PUBLIC               :: do_pwgrid_spherical = 1, &
                                               do_pwgrid_ns_fullspace = 2, &
                                               do_pwgrid_ns_halfspace = 3

   INTEGER, PARAMETER, PUBLIC               :: wfi_use_guess_method_nr = 0, &
                                               wfi_use_prev_p_method_nr = 1, &
                                               wfi_use_prev_rho_r_method_nr = 2, &
                                               wfi_linear_wf_method_nr = 3, &
                                               wfi_linear_p_method_nr = 4, &
                                               wfi_linear_ps_method_nr = 5, &
                                               wfi_use_prev_wf_method_nr = 6, &
                                               wfi_ps_method_nr = 7, &
                                               wfi_frozen_method_nr = 8, &
                                               wfi_aspc_nr = 9

   INTEGER, PARAMETER, PUBLIC               :: do_method_undef = 0, &
                                               do_method_gapw = 1, &
                                               do_method_gapw_xc = 2, &
                                               do_method_gpw = 3, &
                                               do_method_lrigpw = 4, &
                                               do_method_rigpw = 5, &
                                               do_method_mndo = 7, &
                                               do_method_am1 = 8, &
                                               do_method_pm3 = 9, &
                                               do_method_pm6 = 10, &
                                               do_method_mndod = 11, &
                                               do_method_pdg = 12, &
                                               do_method_rm1 = 13, &
                                               do_method_dftb = 14, &
                                               do_method_pchg = 15, &
                                               do_method_pnnl = 16, &
                                               do_method_ofgpw = 17, &
                                               do_method_pm6fm = 18, &
                                               do_method_pw = 19, &
                                               do_method_xtb = 20

   INTEGER, PARAMETER, PUBLIC               :: do_multipole_section_on = 0, &
                                               do_multipole_section_off = 1

   INTEGER, PARAMETER, PUBLIC               :: do_ppl_analytic = 0, &
                                               do_ppl_grid = 1

   INTEGER, PARAMETER, PUBLIC               :: do_se_IS_kdso = 0, &
                                               do_se_IS_kdso_d = 1, &
                                               do_se_IS_slater = 2

   INTEGER, PARAMETER, PUBLIC               :: do_se_lr_none = 0, &
                                               do_se_lr_ewald = 1, &
                                               do_se_lr_ewald_r3 = 2, &
                                               do_se_lr_ewald_gks = 3

   INTEGER, PARAMETER, PUBLIC               :: do_par_atom = 0, &
                                               do_par_grid = 1

   INTEGER, PARAMETER, PUBLIC               :: do_qmmm_center_every_step = 0, &
                                               do_qmmm_center_setup_only = 1, &
                                               do_qmmm_center_never = 2

   INTEGER, PARAMETER, PUBLIC                    :: do_qmmm_center_max_minus_min = 0, &
                                                    do_qmmm_center_pbc_aware = 1
   ! Thermostats
   INTEGER, PARAMETER, PUBLIC               :: do_thermo_no_communication = 0, &
                                               do_thermo_communication = 1, &
                                               do_thermo_only_master = 2

   ! pint coordinate transformations
   INTEGER, PARAMETER, PUBLIC               :: transformation_normal = 1, &
                                               transformation_stage = 2

   ! pint propagator mode
   INTEGER, PARAMETER, PUBLIC               :: propagator_pimd = 1, &
                                               propagator_rpmd = 2, &
                                               propagator_cmd = 3
   ! pint propagator mode
   INTEGER, PARAMETER, PUBLIC               :: integrate_numeric = 1, &
                                               integrate_exact = 2

   ! piglet S-matrix initialisations
   INTEGER, PARAMETER, PUBLIC               :: matrix_init_cholesky = 1, &
                                               matrix_init_diagonal = 2

   ! helium pbc unit cell shapes
   INTEGER, PARAMETER, PUBLIC               :: helium_cell_shape_none = 0, &
                                               helium_cell_shape_cube = 1, &
                                               helium_cell_shape_octahedron = 2

   ! helium-solute interaction potentials
   INTEGER, PARAMETER, PUBLIC               :: helium_solute_intpot_none = 0, &
                                               helium_solute_intpot_mwater = 1, &
                                               helium_solute_intpot_ch5 = 2, &
                                               helium_solute_intpot_runner_io = 3, &
                                               helium_solute_intpot_nnp = 4, &
                                               helium_solute_intpot_cubetable = 5

   ! helium force selection
   INTEGER, PARAMETER, PUBLIC               :: helium_forces_average = 0, &
                                               helium_forces_last = 1
   ! superfluid density, estimators
   INTEGER, PARAMETER, PUBLIC               :: estimator_none = 1, &
                                               estimator_parea = 2, &
                                               estimator_wnumber = 3, &
                                               estimator_pcycle = 4, &
                                               estimator_weighted = 5, &
                                               estimator_normalization = 6

   ! superfluid density, denominator selector
   INTEGER, PARAMETER, PUBLIC               :: denominator_unity = 1, &
                                               denominator_natoms = 2, &
                                               denominator_inertia = 3, &
                                               denominator_rperp2 = 4

   ! helium-sampling algorithms
   INTEGER, PARAMETER, PUBLIC               :: helium_sampling_ceperley = 0, &
                                               helium_sampling_worm = 1

   ! distribution types for sampling path lengths
   INTEGER, PARAMETER, PUBLIC               :: helium_mdist_singlev = 1, &
                                               helium_mdist_uniform = 2, &
                                               helium_mdist_linear = 3, &
                                               helium_mdist_quadratic = 4, &
                                               helium_mdist_exponential = 5, &
                                               helium_mdist_gaussian = 6

   INTEGER, PARAMETER, PUBLIC               :: perm_plain = 1, &
                                               perm_cycle = 2

   ! Free Energy methods
   INTEGER, PARAMETER, PUBLIC               :: do_fe_meta = 0, &
                                               do_fe_ui = 1, &
                                               do_fe_ac = 2

   INTEGER, PARAMETER, PUBLIC               :: do_wall_none = 0, &
                                               do_wall_reflective = 1, &
                                               do_wall_quadratic = 2, &
                                               do_wall_quartic = 3, &
                                               do_wall_gaussian = 4

   INTEGER, PARAMETER, PUBLIC               :: do_wall_p = 0, &
                                               do_wall_m = 1
   ! Output formats
   INTEGER, PARAMETER, PUBLIC               :: dump_atomic = 1, &
                                               dump_xmol = 2, &
                                               dump_dcd = 3, &
                                               dump_pdb = 4, &
                                               dump_dcd_aligned_cell = 5

   INTEGER, PARAMETER, PUBLIC               :: fmt_id_xyz = 1, &
                                               fmt_id_pdb = 2

   INTEGER, PARAMETER, PUBLIC               :: do_qmmm_none = 0, &
                                               do_qmmm_coulomb = 1, &
                                               do_qmmm_gauss = 2, &
                                               do_qmmm_swave = 3, &
                                               do_qmmm_pcharge = 4, &
                                               do_qmmm_link_imomm = 1, &
                                               do_qmmm_link_gho = 2, &
                                               do_qmmm_link_pseudo = 3, &
                                               do_qmmm_center_none = 0, &
                                               do_qmmm_center_grid = 1, &
                                               do_qmmm_wall_none = 0, &
                                               do_qmmm_wall_reflective = 1, &
                                               do_qmmm_wall_quadratic = 2, &
                                               do_qmmm_image_calcmatrix = 0, &
                                               do_qmmm_image_iter = 1

   INTEGER, PARAMETER, PUBLIC               :: calc_always = 0, &
                                               calc_once = 1, &
                                               calc_once_done = 2

   INTEGER, PARAMETER, PUBLIC               :: do_fm_mom_conserv_none = 0, &
                                               do_fm_mom_conserv_equal_f = 1, &
                                               do_fm_mom_conserv_equal_a = 2

   INTEGER, PARAMETER, PUBLIC               :: do_fm_mom_conserv_core = 0, &
                                               do_fm_mom_conserv_qm = 1, &
                                               do_fm_mom_conserv_buffer = 2

   INTEGER, PARAMETER, PUBLIC               :: do_region_global = 0, &
                                               do_region_molecule = 1, &
                                               do_region_massive = 2, &
                                               do_region_none = 3, &
                                               do_region_defined = 4, &
                                               do_bondparm_covalent = 0, &
                                               do_bondparm_vdw = 1, &
                                               do_skip_11 = 0, &
                                               do_skip_12 = 1, &
                                               do_skip_13 = 2, &
                                               do_skip_14 = 3, &
                                               do_conn_psf = 1, &
                                               do_conn_psf_u = 2, &
                                               do_conn_generate = 3, &
                                               do_conn_off = 4, &
                                               do_conn_mol_set = 5, &
                                               do_conn_g87 = 6, &
                                               do_conn_g96 = 7, &
                                               do_conn_amb7 = 8, &
                                               do_conn_user = 9, &
                                               do_coord_off = 0, &
                                               do_coord_pdb = 1, &
                                               do_coord_xyz = 2, &
                                               do_coord_g96 = 3, &
                                               do_coord_crd = 4, &
                                               do_coord_cif = 5, &
                                               do_coord_xtl = 6, &
                                               do_coord_cp2k = 7

   INTEGER, PARAMETER, PUBLIC               :: do_stress_none = 0, &
                                               do_stress_analytical = 1, &
                                               do_stress_numerical = 2, &
                                               do_stress_diagonal_anal = 3, &
                                               do_stress_diagonal_numer = 4

   INTEGER, PARAMETER, PUBLIC               :: do_mc_traditional = 0, &
                                               do_mc_gemc_nvt = 1, &
                                               do_mc_gemc_npt = 2, &
                                               do_mc_virial = 3

   INTEGER, PARAMETER, PUBLIC               :: do_loc_none = 0, &
                                               do_loc_jacobi = 1, &
                                               do_loc_crazy = 2, &
                                               do_loc_direct = 3, &
                                               do_loc_l1_norm_sd = 4, &
                                               do_loc_scdm = 5, &
                                               do_loc_gapo = 6

   INTEGER, PARAMETER, PUBLIC               :: do_loc_cpo_atomic = 0, &
                                               do_loc_cpo_restart = 1, &
                                               do_loc_cpo_random = 2

   INTEGER, PARAMETER, PUBLIC               :: do_loc_cpo_space_wan = 0, &
                                               do_loc_cpo_space_nmo = 1

   INTEGER, PARAMETER, PUBLIC               :: do_loc_min = 0, &
                                               do_loc_max = 1, &
                                               op_loc_berry = 0, &
                                               op_loc_boys = 1, &
                                               op_loc_pipek = 2, &
                                               op_loc_l1_norm = 3, &
                                               state_loc_all = 0, &
                                               state_loc_range = 1, &
                                               state_loc_list = 2, &
                                               energy_loc_range = 3, &
                                               state_loc_none = 4, &
                                               state_loc_mixed = 5

   INTEGER, PARAMETER, PUBLIC               :: do_loc_homo = 0, &
                                               do_loc_lumo = 1, &
                                               do_loc_both = 2, &
                                               do_loc_mixed = 3

   INTEGER, PARAMETER, PUBLIC               :: orb_s = 0, &
                                               orb_px = 1, &
                                               orb_py = 2, &
                                               orb_pz = 3, &
                                               orb_dxy = 4, &
                                               orb_dyz = 5, &
                                               orb_dzx = 6, &
                                               orb_dx2 = 7, &
                                               orb_dy2 = 8, &
                                               orb_dz2 = 9

   INTEGER, PARAMETER, PUBLIC               :: use_mom_ref_com = 0, &
                                               use_mom_ref_coac = 1, &
                                               use_mom_ref_user = 2, &
                                               use_mom_ref_zero = 3

   INTEGER, PARAMETER, PUBLIC               :: current_gauge_r = 1, &
                                               current_gauge_r_and_step_func = 2, &
                                               current_gauge_atom = 3, &
                                               current_orb_center_wannier = 1, &
                                               current_orb_center_common = 2, &
                                               current_orb_center_atom = 3, &
                                               current_orb_center_box = 4

   INTEGER, PARAMETER, PUBLIC               :: ot_mini_sd = 1, ot_mini_cg = 2, ot_mini_diis = 3, ot_mini_broyden = 4
   INTEGER, PARAMETER, PUBLIC               :: ot_algo_taylor_or_diag = 1, ot_algo_irac = 2
   INTEGER, PARAMETER, PUBLIC               :: ot_chol_irac = 1, ot_poly_irac = 2, ot_lwdn_irac = 3
   INTEGER, PARAMETER, PUBLIC               :: ls_none = 1, ls_2pnt = 2, ls_3pnt = 3, ls_gold = 4, ls_fit = 5
   INTEGER, PARAMETER, PUBLIC               :: ot_precond_none = 0, &
                                               ot_precond_full_single = 1, &
                                               ot_precond_full_kinetic = 2, &
                                               ot_precond_s_inverse = 3, &
                                               ot_precond_full_all = 5, &
                                               ot_precond_full_single_inverse = 6, &
                                               precond_mlp = 7
   INTEGER, PARAMETER, PUBLIC               :: ot_precond_solver_default = 1, &
                                               ot_precond_solver_inv_chol = 2, &
                                               ot_precond_solver_direct = 3, &
                                               ot_precond_solver_update = 4

   INTEGER, PARAMETER, PUBLIC               :: diag_standard = 1, &
                                               diag_ot = 2, &
                                               diag_block_krylov = 3, &
                                               diag_block_davidson = 4, &
                                               diag_filter_matrix = 5

   INTEGER, PARAMETER, PUBLIC               :: default_lbfgs_method_id = 1, &
                                               default_bfgs_method_id = 2, &
                                               default_cg_method_id = 3

   INTEGER, PARAMETER, PUBLIC               :: default_minimization_method_id = 1, &
                                               default_ts_method_id = 2, &
                                               default_cell_method_id = 3, &
                                               default_shellcore_method_id = 4

   INTEGER, PARAMETER, PUBLIC               :: none_ts_method_id = 0, &
                                               default_dimer_method_id = 1

   INTEGER, PARAMETER, PUBLIC               :: do_first_rotation_step = 0, &
                                               do_second_rotation_step = 1, &
                                               do_third_rotation_step = 2

   INTEGER, PARAMETER, PUBLIC               :: default_cell_geo_opt_id = 0, &
                                               default_cell_md_id = 1, &
                                               default_cell_direct_id = 2

   INTEGER, PARAMETER, PUBLIC               :: xc_funct_no_shortcut = 0, &
                                               xc_funct_blyp = 1, &
                                               xc_funct_pade = 2, &
                                               xc_funct_pbe = 3, &
                                               xc_funct_tpss = 4, &
                                               xc_funct_hcth120 = 5, &
                                               xc_funct_olyp = 6, &
                                               xc_funct_bp = 7, &
                                               xc_none = 8, &
                                               xc_funct_xwpbe = 9, &
                                               xc_funct_b3lyp = 10, &
                                               xc_funct_pbe0 = 11, &
                                               xc_funct_beefvdw = 12
   INTEGER, PARAMETER, PUBLIC               :: fxc_none = 0, &
                                               fxc_funct_pade = 1, &
                                               fxc_funct_lda = 2, &
                                               fxc_funct_gga = 3
   INTEGER, PARAMETER, PUBLIC               :: sic_none = 0, &
                                               sic_mauri_us = 1, &
                                               sic_mauri_spz = 2, &
                                               sic_ad = 3, &
                                               sic_eo = 4
   INTEGER, PARAMETER, PUBLIC               :: sic_list_all = 1, &
                                               sic_list_unpaired = 2
   INTEGER, PARAMETER, PUBLIC               :: tddfpt_singlet = 0, &
                                               tddfpt_triplet = 1, &
                                               tddfpt_spin_cons = 2, &
                                               tddfpt_spin_flip = 3
   INTEGER, PARAMETER, PUBLIC               :: tddfpt_lanczos = 0, &
                                               tddfpt_davidson = 1
   INTEGER, PARAMETER, PUBLIC               :: tddfpt_kernel_none = 2, &
                                               tddfpt_kernel_full = 1, &
                                               tddfpt_kernel_stda = 0
   INTEGER, PARAMETER, PUBLIC               :: oe_none = 0, &
                                               oe_lb = 1, &
                                               oe_gllb = 2, &
                                               oe_saop = 3, &
                                               oe_sic = 44, &
                                               oe_shift = 4
   INTEGER, PARAMETER, PUBLIC               :: no_excitations = 0, &
                                               tddfpt_excitations = 1
   INTEGER, PARAMETER, PUBLIC               :: xc_vdw_fun_none = 100, &
                                               xc_vdw_fun_pairpot = 101, &
                                               xc_vdw_fun_nonloc = 102
   INTEGER, PARAMETER, PUBLIC               :: vdw_pairpot_dftd2 = 1, &
                                               vdw_pairpot_dftd3 = 2, &
                                               vdw_pairpot_dftd3bj = 3, &
                                               vdw_pairpot_dftd4 = 4
   INTEGER, PARAMETER, PUBLIC               :: vdw_nl_DRSLL = 1, &
                                               vdw_nl_LMKLL = 2, &
                                               vdw_nl_RVV10 = 3

   ! Input constants for relativistic calculations
   INTEGER, PARAMETER, PUBLIC               :: rel_none = 0, &
                                               rel_dkh = 1, &
                                               rel_zora = 2
   INTEGER, PARAMETER, PUBLIC               :: rel_zora_full = 100, &
                                               rel_zora_mp = 101, &
                                               rel_sczora_mp = 102
   INTEGER, PARAMETER, PUBLIC               :: rel_trans_full = 0, &
                                               rel_trans_molecule = 1, &
                                               rel_trans_atom = 2
   INTEGER, PARAMETER, PUBLIC               :: rel_pot_full = 0, &
                                               rel_pot_erfc = 1

   INTEGER, PARAMETER, PUBLIC               :: do_add = 1, &
                                               do_remove = 2

   INTEGER, PARAMETER, PUBLIC               :: xc_pot_energy_none = 0, &
                                               xc_pot_energy_xc_functional = 1, &
                                               xc_pot_energy_sum_eigenvalues = 2

   INTEGER, PARAMETER, PUBLIC               :: xc_pot_none = 1000, &
                                               xc_pot_saop = 1001

   INTEGER, PARAMETER, PUBLIC               :: xtb_vdw_type_none = 0, &
                                               xtb_vdw_type_d3 = 1, &
                                               xtb_vdw_type_d4 = 2

   ! Hirshfeld partitioning
   INTEGER, PARAMETER, PUBLIC               :: shape_function_gaussian = 1, &
                                               shape_function_density = 2
   INTEGER, PARAMETER, PUBLIC               :: ref_charge_atomic = 100, &
                                               ref_charge_mulliken = 200
   INTEGER, PARAMETER, PUBLIC               :: radius_covalent = 10, &
                                               radius_user = 11, &
                                               radius_single = 12, &
                                               radius_vdw = 13, &
                                               radius_default = 14

   ! Voronoi integration
   INTEGER, PARAMETER, PUBLIC               :: voro_radii_unity = 100, &
                                               voro_radii_vdw = 101, &
                                               voro_radii_cov = 102, &
                                               voro_radii_user = 103

   ! BQB Format
   INTEGER, PARAMETER, PUBLIC               :: bqb_opt_off = 100, &
                                               bqb_opt_quick = 101, &
                                               bqb_opt_normal = 102, &
                                               bqb_opt_patient = 103, &
                                               bqb_opt_exhaustive = 104

   ! MAO
   INTEGER, PARAMETER, PUBLIC               :: mao_basis_orb = 2000, &
                                               mao_basis_prim = 2001, &
                                               mao_basis_ext = 2002

   ! IAO
   INTEGER, PARAMETER, PUBLIC               :: do_iaoloc_pm2 = 100, &
                                               do_iaoloc_pm4 = 200, &
                                               do_iaoloc_l1 = 300, &
                                               do_iaoloc_enone = 1000, &
                                               do_iaoloc_energy = 1001, &
                                               do_iaoloc_occ = 1002

   ! X-Ray spectra Calculation
   INTEGER, PARAMETER, PUBLIC               :: xas_none = 0, &
                                               xas_tp_hh = 1, &
                                               xas_tp_fh = 2, &
                                               xes_tp_val = 3, &
                                               xas_tp_xhh = 4, &
                                               xas_tp_xfh = 5, &
                                               xas_dscf = 6, &
                                               xas_tp_flex = 7
   INTEGER, PARAMETER, PUBLIC               :: xas_not_excited = 0, &
                                               xas_1s_type = 1, &
                                               xas_2s_type = 2, &
                                               xas_2p_type = 3, &
                                               xas_3s_type = 4, &
                                               xas_3p_type = 5, &
                                               xas_3d_type = 6, &
                                               xas_4s_type = 7, &
                                               xas_4p_type = 8, &
                                               xas_4d_type = 9, &
                                               xas_4f_type = 10
   INTEGER, PARAMETER, PUBLIC               :: xas_dip_len = 1, &
                                               xas_dip_vel = 2
   INTEGER, PARAMETER, PUBLIC               :: xas_scf_default = 0, &
                                               xas_scf_general = 1

   ! Time-dependent XAS
   INTEGER, PARAMETER, PUBLIC               :: xas_tdp_by_index = 1, &
                                               xas_tdp_by_kind = 2, &
                                               xas_tdp_roks = 1, &
                                               xas_tdp_uks = 2

   ! Form of dipole operator for TDDFPT oscillator strength calculation
   INTEGER, PARAMETER, PUBLIC               :: tddfpt_dipole_berry = 1, &
                                               tddfpt_dipole_length = 2, &
                                               tddfpt_dipole_velocity = 3

   ! XC Kernel derivative methods for forces
   INTEGER, PARAMETER, PUBLIC               :: xc_kernel_method_best = 100, &
                                               xc_kernel_method_analytic = 101, &
                                               xc_kernel_method_numeric = 102

   ! Linear Response for properties
   INTEGER, PARAMETER, PUBLIC               :: lr_none = 0, &
                                               lr_chemshift = 1, &
                                               lr_gtensor = 2, &
                                               lr_current = 3

   ! EIP models
   INTEGER, PARAMETER, PUBLIC               :: use_bazant_eip = 1, &
                                               use_lenosky_eip = 2

   ! ddapc restraint forms
   INTEGER, PARAMETER, PUBLIC               :: do_ddapc_restraint = 773, &
                                               do_ddapc_constraint = 774

   ! outer scf types
   INTEGER, PARAMETER, PUBLIC               :: outer_scf_ddapc_constraint = 123, &
                                               outer_scf_s2_constraint = 124, &
                                               outer_scf_becke_constraint = 125, &
                                               outer_scf_none = 126, &
                                               outer_scf_basis_center_opt = 127, &
                                               outer_scf_cdft_constraint = 128, &
                                               outer_scf_hirshfeld_constraint = 129

   ! outer scf optimizers
   INTEGER, PARAMETER, PUBLIC               :: outer_scf_optimizer_sd = 1001, &
                                               outer_scf_optimizer_diis = 1002, &
                                               outer_scf_optimizer_none = 1003, &
                                               outer_scf_optimizer_bisect = 1004, &
                                               outer_scf_optimizer_broyden = 1005, &
                                               outer_scf_optimizer_newton = 1006, &
                                               outer_scf_optimizer_secant = 1007, &
                                               outer_scf_optimizer_newton_ls = 1008

   ! outer scf broyden optimizer types
   INTEGER, PARAMETER, PUBLIC               :: broyden_type_1 = 1101, &
                                               broyden_type_1_explicit = 1102, &
                                               broyden_type_2 = 1103, &
                                               broyden_type_2_explicit = 1104, &
                                               broyden_type_1_ls = 1105, &
                                               broyden_type_1_explicit_ls = 1106, &
                                               broyden_type_2_ls = 1107, &
                                               broyden_type_2_explicit_ls = 1108

   ! finite difference types for calculation of inverse jacobian
   INTEGER, PARAMETER, PUBLIC               :: jacobian_fd1 = 1, &
                                               jacobian_fd1_backward = 2, &
                                               jacobian_fd2 = 3, &
                                               jacobian_fd2_backward = 4, &
                                               jacobian_fd1_central = 5

   ! s2 restraint forms
   INTEGER, PARAMETER, PUBLIC               :: do_s2_restraint = 872, &
                                               do_s2_constraint = 873

   ! Becke cutoff schemes
   INTEGER, PARAMETER, PUBLIC               :: becke_cutoff_global = 790, &
                                               becke_cutoff_element = 791

   ! CDFT constraint and control types
   INTEGER, PARAMETER, PUBLIC               :: ot2cdft = 101, &
                                               cdft2ot = 102

   ! CDFT constraint type definitions
   INTEGER, PARAMETER, PUBLIC               :: cdft_charge_constraint = 820, &
                                               cdft_magnetization_constraint = 821, &
                                               cdft_alpha_constraint = 822, &
                                               cdft_beta_constraint = 823

   ! Mixed CDFT calculation run types
   INTEGER, PARAMETER, PUBLIC               :: mixed_cdft_serial = 90, &
                                               mixed_cdft_parallel_nobuild = 91, &
                                               mixed_cdft_parallel = 92

   ! ROKS schemes
   INTEGER, PARAMETER, PUBLIC               :: general_roks = 1, &
                                               high_spin_roks = 2

   ! mixing of force envs
   INTEGER, PARAMETER, PUBLIC               :: mix_linear_combination = 701, &
                                               mix_minimum = 702, &
                                               mix_coupled = 703, &
                                               mix_restrained = 704, &
                                               mix_generic = 705, &
                                               mix_mymix = 706, &
                                               mix_cdft = 707

   ! cost models for distribution 2d
   INTEGER, PARAMETER, PUBLIC               :: model_block_count = 453, &
                                               model_block_surface = 454, &
                                               model_block_lmax = 455

   ! clustering methods
   INTEGER, PARAMETER, PUBLIC               :: clustering_none = 0, &
                                               clustering_box = 1, &
                                               clustering_h_on_heavier = 2, &
                                               clustering_small_on_larger = 3

   ! HFX potential type (ri_default for RI but used with potentials)
   INTEGER, PARAMETER, PUBLIC               :: do_potential_coulomb = 1, &
                                               do_potential_short = 2, &
                                               do_potential_long = 3, &
                                               do_potential_mix_cl = 4, &
                                               do_potential_gaussian = 5, &
                                               do_potential_mix_lg = 6, &
                                               do_potential_id = 7, &
                                               do_potential_truncated = 8, &
                                               do_potential_mix_cl_trunc = 9, &
                                               do_potential_TShPSC = 1009, &
                                               ri_default = 10001

   ! HFX periodic number of shells
   INTEGER, PARAMETER, PUBLIC               :: do_hfx_auto_shells = -1

   ! adiabatic hybrid type
   INTEGER, PARAMETER, PUBLIC               :: do_adiabatic_hybrid_mcy3 = 1

   ! adiabatic model type
   INTEGER, PARAMETER, PUBLIC               :: do_adiabatic_model_pade = 1

   ! HFX evaluation type
   INTEGER, PARAMETER, PUBLIC               :: hfx_do_eval_energy = 1, &
                                               hfx_do_eval_forces = 2

   ! HFX RI matrix methods
   INTEGER, PARAMETER, PUBLIC               :: hfx_ri_do_2c_iter = 1, &
                                               hfx_ri_do_2c_diag = 2, &
                                               hfx_ri_do_2c_cholesky = 3

   ! mode selctive vibrational analysis
   INTEGER, PARAMETER, PUBLIC               :: ms_guess_bfgs = 1, &
                                               ms_guess_atomic = 2, &
                                               ms_guess_restart = 3, &
                                               ms_guess_restart_vec = 4, &
                                               ms_guess_molden = 5

   ! BASIS SET PARAMETER
   ! This parameter identifies basis sets
   INTEGER, PARAMETER, PUBLIC               :: use_orb_basis_set = 1, &
                                               use_aux_fit_basis_set = 2, &
                                               use_aux_basis_set = 3, &
                                               use_ri_aux_basis_set = 4, &
                                               use_lri_basis_set = 5, &
                                               use_p_lri_basis_set = 6
   ! ADMM PARAMETER
   ! This parameter defines the method to use for the purification
   INTEGER, PARAMETER, PUBLIC               :: do_admm_purify_none = 1, &
                                               do_admm_purify_cauchy = 2, &
                                               do_admm_purify_cauchy_subspace = 3, &
                                               do_admm_purify_mo_diag = 4, &
                                               do_admm_purify_mo_no_diag = 5, &
                                               do_admm_purify_mcweeny = 6, &
                                               do_admm_purify_none_dm = 7

   ! This parameter defines when to use blocking
   INTEGER, PARAMETER, PUBLIC               :: do_admm_basis_projection = 0, &
                                               do_admm_blocking_purify_full = 1, &
                                               do_admm_blocked_projection = 2, &
                                               do_admm_charge_constrained_projection = 3

   ! This parameter defines when to use blocking
   INTEGER, PARAMETER, PUBLIC               :: do_admm_aux_exch_func_default = 0, &
                                               do_admm_aux_exch_func_pbex = 1, &
                                               do_admm_aux_exch_func_none = 2, &
                                               do_admm_aux_exch_func_opt = 3, &
                                               do_admm_aux_exch_func_bee = 4, &
                                               do_admm_aux_exch_func_default_libxc = 5, &
                                               do_admm_aux_exch_func_pbex_libxc = 6, &
                                               do_admm_aux_exch_func_opt_libxc = 7, &
                                               do_admm_aux_exch_func_bee_libxc = 8, &
                                               do_admm_aux_exch_func_sx_libxc = 9

   ! Define scaling of the exchange correction
   INTEGER, PARAMETER, PUBLIC               :: do_admm_exch_scaling_none = 0, &
                                               do_admm_exch_scaling_merlot = 1
   ! ADMM TYPES
   INTEGER, PARAMETER, PUBLIC               :: no_admm_type = 0, &
                                               admm1_type = 1, &
                                               admm2_type = 2, &
                                               admms_type = 3, &
                                               admmp_type = 4, &
                                               admmq_type = 5

   ! Define DFT+U method identifier codes
   INTEGER, PARAMETER, PUBLIC               :: plus_u_lowdin = 1, &
                                               plus_u_mulliken = 2, &
                                               plus_u_mulliken_charges = 3

   ! Define identifier codes for smearing method
   INTEGER, PARAMETER, PUBLIC               :: smear_energy_window = 1, &
                                               smear_fermi_dirac = 2, &
                                               smear_list = 3

   ! Define identifier codes for mixing method

   INTEGER, PARAMETER, PUBLIC               :: no_mix = 0, &
                                               direct_p_mix = 1, &
                                               kerker_mix = 2, &
                                               pulay_mix = 3, &
                                               broy_mix = 4, &
                                               broy_mix_new = 5, &
                                               multisec_mix = 6

   ! FFT library
   ! these might need sync with fft_lib.F
   INTEGER, PARAMETER, PUBLIC               :: do_fft_sg = 1, &
                                               do_fft_fftw3 = 3
   INTEGER, PARAMETER, PUBLIC               :: fftw_plan_estimate = 1, &
                                               fftw_plan_measure = 2, &
                                               fftw_plan_patient = 3, &
                                               fftw_plan_exhaustive = 4
   ! Local DGEMM library
   INTEGER, PARAMETER, PUBLIC               :: do_dgemm_spla = 1, &
                                               do_dgemm_blas = 2

   !real time propagation
   INTEGER, PARAMETER, PUBLIC               :: do_taylor = 1, &
                                               do_pade = 2, &
                                               do_arnoldi = 3, &
                                               do_bch = 4, &
                                               do_exact = 5

   INTEGER, PARAMETER, PUBLIC               :: do_etrs = 1, &
                                               do_cn = 2, &
                                               do_em = 3

   INTEGER, PARAMETER, PUBLIC               :: use_scf_wfn = 1, &
                                               use_restart_wfn = 2, &
                                               use_rt_restart = 3

   INTEGER, PARAMETER, PUBLIC               :: constant_env = 1, &
                                               gaussian_env = 2, &
                                               ramp_env = 3, &
                                               custom_env = 4

   INTEGER, PARAMETER, PUBLIC               :: rtp_method_tddft = 1, &
                                               rtp_method_bse = 2

   INTEGER, PARAMETER, PUBLIC               :: rtp_bse_ham_ks = 1, &
                                               rtp_bse_ham_g0w0 = 2

   ! type of reference for RTP%PRINT%PROJECTION_MO
   INTEGER, PARAMETER, PUBLIC               :: proj_mo_ref_scf = 1, &
                                               proj_mo_ref_xas_tdp = 2

   ! how to solve polarizable force fields
   INTEGER, PARAMETER, PUBLIC               :: do_fist_pol_none = 1, &
                                               do_fist_pol_sc = 2, &
                                               do_fist_pol_cg = 3

   ! some dm_ls_scf parameters
   INTEGER, PARAMETER, PUBLIC               :: ls_s_preconditioner_none = 7, &
                                               ls_s_preconditioner_atomic = 13, &
                                               ls_s_preconditioner_molecular = 17

   INTEGER, PARAMETER, PUBLIC               :: ls_cluster_atomic = 111, &
                                               ls_cluster_molecular = 112

   INTEGER, PARAMETER, PUBLIC               :: ls_s_inversion_hotelling = 3, &
                                               ls_s_inversion_sign_sqrt = 5, &
                                               ls_s_inversion_none = 6

   INTEGER, PARAMETER, PUBLIC               :: ls_s_sqrt_ns = 1, &
                                               ls_s_sqrt_proot = 2

   INTEGER, PARAMETER, PUBLIC               :: ls_scf_sign = 17, ls_scf_trs4 = 18, &
                                               ls_scf_tc2 = 19, ls_scf_pexsi = 20

   INTEGER, PARAMETER, PUBLIC               :: ls_scf_sign_ns = 1, ls_scf_sign_proot = 2, ls_scf_sign_submatrix = 3

   INTEGER, PARAMETER, PUBLIC               :: ls_scf_submatrix_sign_ns = 1, ls_scf_submatrix_sign_direct = 2, &
                                               ls_scf_submatrix_sign_direct_muadj = 3, ls_scf_submatrix_sign_direct_muadj_lowmem = 4

   INTEGER, PARAMETER, PUBLIC               :: ls_scf_line_search_3point = 3, &
                                               ls_scf_line_search_3point_2d = 6
   ! some ZMP paramenters
   INTEGER, PARAMETER, PUBLIC               :: use_coulomb = 1, &
                                               use_diff = 2, &
                                               use_no = 3

   ! parameters for ALMO methods
   INTEGER, PARAMETER, PUBLIC   :: almo_domain_layout_orbital = 1, &
                                   almo_domain_layout_atomic = 10, &
                                   almo_domain_layout_molecular = 100

   INTEGER, PARAMETER, PUBLIC   :: almo_mat_distr_atomic = 1, &
                                   almo_mat_distr_molecular = 2

   INTEGER, PARAMETER, PUBLIC   :: almo_constraint_block_diagonal = 0, &
                                   almo_constraint_distance = 1, &
                                   almo_constraint_ao_overlap = 2

   ! methods to treat electron delocalization
   INTEGER, PARAMETER, PUBLIC   :: almo_deloc_none = 0, &
                                   almo_deloc_x = 10, &
                                   almo_deloc_xalmo_x = 11, &
                                   almo_deloc_xk = 12, &
                                   almo_deloc_xalmo_scf = 13, &
                                   almo_deloc_xalmo_1diag = 14, &
                                   almo_deloc_scf = 100, &
                                   almo_deloc_x_then_scf = 110

   INTEGER, PARAMETER, PUBLIC   :: almo_scf_dm_sign = 1, &
                                   almo_scf_diag = 2, &
                                   almo_scf_pcg = 3, &
                                   almo_scf_trustr = 4, &
                                   almo_scf_skip = 0

   INTEGER, PARAMETER, PUBLIC   :: penalty_type_none = 0, &
                                   penalty_type_lndet = 1, &
                                   penalty_type_nlmo = 2

   ! optimizer parameters
   INTEGER, PARAMETER, PUBLIC   :: cg_zero = 0, &
                                   cg_polak_ribiere = 1, &
                                   cg_fletcher_reeves = 2, &
                                   cg_hestenes_stiefel = 3, &
                                   cg_fletcher = 4, &
                                   cg_liu_storey = 5, &
                                   cg_dai_yuan = 6, &
                                   cg_hager_zhang = 7
   INTEGER, PARAMETER, PUBLIC   :: trustr_steihaug = 1, &
                                   trustr_cauchy = 2, &
                                   trustr_dogleg = 3
   INTEGER, PARAMETER, PUBLIC   :: optimizer_diis = 1, &
                                   optimizer_pcg = 2, &
                                   optimizer_trustr = 4, &
                                   optimizer_lin_eq_pcg = 3
   INTEGER, PARAMETER, PUBLIC   :: xalmo_prec_zero = 0, &
                                   xalmo_prec_domain = 1, &
                                   xalmo_prec_full = 2
   INTEGER, PARAMETER, PUBLIC   :: xalmo_case_block_diag = 0, &
                                   xalmo_case_fully_deloc = 1, &
                                   xalmo_case_normal = -1
   INTEGER, PARAMETER, PUBLIC   :: xalmo_trial_simplex = 0, &
                                   xalmo_trial_r0_out = 1

   ! parameters for CT methods
   INTEGER, PARAMETER, PUBLIC   :: tensor_orthogonal = 1, &
                                   tensor_up_down = 2
   INTEGER, PARAMETER, PUBLIC   :: virt_full = 1, &
                                   virt_minimal = 2, &
                                   virt_occ_size = 3, &
                                   virt_number = 4

   ! spd matrix inversion algorithm
   INTEGER, PARAMETER, PUBLIC   :: spd_inversion_ls_hotelling = 0, &
                                   spd_inversion_dense_cholesky = 1, &
                                   spd_inversion_ls_taylor = 2

   ! some MP2 parameters
   INTEGER, PARAMETER, PUBLIC               :: mp2_method_none = 0, &
                                               mp2_method_direct = 4, &
                                               mp2_method_gpw = 6, &
                                               ri_mp2_method_gpw = 7, &
                                               ri_rpa_method_gpw = 8, &
                                               ri_mp2_laplace = 9, &
                                               mp2_ri_optimize_basis = 10, &
                                               wfc_mm_style_gemm = 11, &
                                               wfc_mm_style_syrk = 12

   ! RPA exchange corrections
   INTEGER, PARAMETER, PUBLIC               :: rpa_exchange_none = 0, &
                                               rpa_exchange_axk = 1, rpa_exchange_sosex = 2

   ! solvers of the z-vector equations
   INTEGER, PARAMETER, PUBLIC               :: z_solver_pople = 1, &
                                               z_solver_cg = 2, &
                                               z_solver_richardson = 3, &
                                               z_solver_sd = 4

   ! GW parameter
   INTEGER, PARAMETER, PUBLIC               :: ri_rpa_g0w0_crossing_z_shot = 1, &
                                               ri_rpa_g0w0_crossing_newton = 2, &
                                               ri_rpa_g0w0_crossing_bisection = 3, &
                                               gw_no_print_exx = 5, &
                                               gw_print_exx = 6, &
                                               gw_read_exx = 7, &
                                               gw_skip_for_regtest = 8, &
                                               gw_pade_approx = 9, &
                                               gw_two_pole_model = 10, &
                                               kp_weights_W_auto = 11, &
                                               kp_weights_W_uniform = 12, &
                                               kp_weights_W_tailored = 13, &
                                               gw_gf_mic = 14, &
                                               gw_gf_gamma = 15, &
                                               soc_none = 16, &
                                               soc_lda = 17, &
                                               soc_pbe = 18, &
                                               int_ldos_x = 19, &
                                               int_ldos_y = 20, &
                                               int_ldos_z = 21, &
                                               int_ldos_none = 22, &
                                               small_cell_full_kp = 31, &
                                               large_cell_Gamma = 32, &
                                               G0W0 = 35, &
                                               evGW0 = 36, &
                                               evGW = 37

   ! periodic RESP parameters
   INTEGER, PARAMETER, PUBLIC               :: do_resp_x_dir = 0, &
                                               do_resp_y_dir = 1, &
                                               do_resp_z_dir = 2, &
                                               do_resp_minus_x_dir = 3, &
                                               do_resp_minus_y_dir = 4, &
                                               do_resp_minus_z_dir = 5, &
                                               use_cambridge_vdw_radii = 0, &
                                               use_uff_vdw_radii = 1

   ! basis optimization parameters
   INTEGER, PARAMETER, PUBLIC               :: do_opt_all = 0, &
                                               do_opt_none = 1, &
                                               do_opt_coeff = 2, &
                                               do_opt_exps = 3

   ! LRI parameters for inverse of overlap
   INTEGER, PARAMETER, PUBLIC               :: do_lri_inv = 0, &
                                               do_lri_pseudoinv_svd = 1, &
                                               do_lri_pseudoinv_diag = 2, &
                                               do_lri_inv_auto = 3

   ! LRI basis optimization parameters
   INTEGER, PARAMETER, PUBLIC               :: do_lri_opt_all = 0, &
                                               do_lri_opt_coeff = 1, &
                                               do_lri_opt_exps = 2

   ! Active space model parameters
   INTEGER, PARAMETER, PUBLIC               :: hf_model = 100, &
                                               rsdft_model = 101, &
                                               dmft_model = 102

   ! Active space external solvers
   INTEGER, PARAMETER, PUBLIC               :: no_solver = 200, &
                                               qiskit_solver = 201

   ! callgraph parameters
   INTEGER, PARAMETER, PUBLIC               :: callgraph_none = 0, &
                                               callgraph_master = 1, &
                                               callgraph_all = 2

   ! kg coloring parameters
   INTEGER, PARAMETER, PUBLIC               :: kg_color_dsatur = 0, &
                                               kg_color_greedy = 1
   ! kg kinetic energy embedding potential
   INTEGER, PARAMETER, PUBLIC               :: kg_tnadd_embed = 100, &
                                               kg_tnadd_embed_ri = 101, &
                                               kg_tnadd_atomic = 200, &
                                               kg_tnadd_none = 300

   INTEGER, PARAMETER, PUBLIC               :: kg_cholesky = 3001

   ! Harris method
   INTEGER, PARAMETER, PUBLIC               :: hfun_harris = 1
   INTEGER, PARAMETER, PUBLIC               :: hden_atomic = 10
   INTEGER, PARAMETER, PUBLIC               :: horb_default = 100

   ! non-scf energy corrections
   INTEGER, PARAMETER, PUBLIC               :: ec_functional_harris = 2001, &
                                               ec_functional_dc = 2002, &
                                               ec_functional_ext = 2003

   ! Energy correction solver
   INTEGER, PARAMETER, PUBLIC               :: ec_diagonalization = 1001, &
                                               ec_curvy_steps = 1002, &
                                               ec_matrix_sign = 1003, &
                                               ec_matrix_trs4 = 1004, &
                                               ec_matrix_tc2 = 1005, &
                                               ec_ot_diag = 1006

   ! response solver for energy correction
   INTEGER, PARAMETER, PUBLIC               :: ec_ot_atomic = 1, &
                                               ec_ot_gs = 2

   ! response solver for energy correction
   INTEGER, PARAMETER, PUBLIC               :: ec_mo_solver = 1, &
                                               ec_ls_solver = 2

   ! swarm parameters
   INTEGER, PARAMETER, PUBLIC               :: swarm_do_glbopt = 1

   ! global opt parameters
   INTEGER, PARAMETER, PUBLIC               :: glbopt_do_minhop = 1, &
                                               glbopt_do_mincrawl = 2

   ! Self-consistent continuum solvation (SCCS) models
   INTEGER, PARAMETER, PUBLIC               :: sccs_andreussi = 0, &
                                               sccs_fattebert_gygi = 1

   ! Derivative method used by the SCCS
   INTEGER, PARAMETER, PUBLIC               :: sccs_derivative_fft = 0, &
                                               sccs_derivative_cd3 = 1, &
                                               sccs_derivative_cd5 = 2, &
                                               sccs_derivative_cd7 = 3

   ! fm matrix multiplication
   INTEGER, PARAMETER, PUBLIC               :: do_scalapack = 1, &
                                               do_cosma = 2

   ! Dispersion DFTB
   INTEGER, PARAMETER, PUBLIC               :: dispersion_none = 0, &
                                               dispersion_uff = 100, &
                                               dispersion_d3 = 200, &
                                               dispersion_d3bj = 210, &
                                               dispersion_d2 = 300

   ! Transport section
   INTEGER, PARAMETER, PUBLIC               :: transport_localscf = 1, &
                                               transport_transmission = 2, &
                                               transport_negf = 3, &
                                               linsolver_splitsolve = 11, &
                                               linsolver_superlu = 12, &
                                               linsolver_mumps = 13, &
                                               linsolver_full = 14, &
                                               linsolver_banded = 15, &
                                               linsolver_pardiso = 16, &
                                               linsolver_umfpack = 17, &
                                               matrixinv_full = 101, &
                                               matrixinv_pexsi = 102, &
                                               matrixinv_pardiso = 103, &
                                               matrixinv_rgf = 104, &
                                               injmethod_evp = 21, &
                                               injmethod_beyn = 22, &
                                               injsign_positive = 1, &
                                               injsign_negative = -1, &
                                               rlaxisint_GaussChebyshev = 31, &
                                               rlaxisint_trapezoidal = 32, &
                                               rlaxisint_readfromfile = 33, &
                                               rho_negf = 41, &
                                               rho_qtbm = 42, &
                                               neutlead_bs = 51, &
                                               neutlead_dos = 52

   ! Active Space Section: orbital selection methods
   INTEGER, PARAMETER, PUBLIC               :: casci_canonical = 100, &
                                               wannier_projection = 101, &
                                               mao_projection = 102, &
                                               manual_selection = 103
   INTEGER, PARAMETER, PUBLIC               :: eri_method_full_gpw = 1, &
                                               eri_method_gpw_ht = 2
   INTEGER, PARAMETER, PUBLIC               :: eri_operator_coulomb = 1, &
                                               eri_operator_yukawa = 2, &
                                               eri_operator_erf = 3, &
                                               eri_operator_erfc = 4, &
                                               eri_operator_gaussian = 5, &
                                               eri_operator_trunc = 6

   INTEGER, PARAMETER, PUBLIC               :: do_eri_gpw = 0, &
                                               do_eri_mme = 1, &
                                               do_eri_os = 2, &
                                               eri_default = 3

   ! Cell optimisation constraints
   INTEGER, PARAMETER, PUBLIC               :: fix_none = 0, &
                                               fix_x = 1, &
                                               fix_y = 2, &
                                               fix_z = 3, &
                                               fix_xy = 4, &
                                               fix_xz = 5, &
                                               fix_yz = 6

   ! Non-equilibrium Green's function method
   INTEGER, PARAMETER, PUBLIC               :: negfint_method_cc = 0, &
                                               negfint_method_simpson = 1
   INTEGER, PARAMETER, PUBLIC               :: negfrun_main = 0, &
                                               negfrun_fermilevel = 1
   ! Type of embedding
   INTEGER, PARAMETER, PUBLIC               :: dfet = 0, &
                                               dmfet = 1

   ! DFT embedding: optimization of potential
   INTEGER, PARAMETER, PUBLIC               :: embed_steep_desc = 0, &
                                               embed_quasi_newton = 1, &
                                               embed_level_shift = 2
   INTEGER, PARAMETER, PUBLIC               :: embed_grid_bohr = 0, &
                                               embed_grid_angstrom = 1
   INTEGER, PARAMETER, PUBLIC               :: embed_none = 0, &
                                               embed_diff = 1, &
                                               embed_fa = 2, &
                                               embed_resp = 3

   ! MOLDEN format
   INTEGER, PARAMETER, PUBLIC               :: gto_cartesian = 1, &
                                               gto_spherical = 2

   ! Tip scan directions
   INTEGER, PARAMETER, PUBLIC               :: scan_x = 1, &
                                               scan_y = 2, &
                                               scan_z = 3, &
                                               scan_xy = 12, &
                                               scan_xz = 13, &
                                               scan_yz = 23, &
                                               scan_xyz = 123

   ! DFT%PRINT%WFN_MIX orgine type
   INTEGER, PARAMETER, PUBLIC               :: wfn_mix_orig_occ = 1, &
                                               wfn_mix_orig_virtual = 2, &
                                               wfn_mix_orig_external = 3

   ! DFT%PRINT%E_DENSITY_CUBE density type
   INTEGER, PARAMETER, PUBLIC               :: e_dens_total_hard_approx = 1, &
                                               e_dens_total_density = 2, &
                                               e_dens_soft_density = 3
   ! BSE spin configuration
   INTEGER, PARAMETER, PUBLIC               :: bse_singlet = 0, &
                                               bse_triplet = 1

   ! BSE method of diagonalization (full diagonalization or iteratively)
   INTEGER, PARAMETER, PUBLIC               :: bse_fulldiag = 0, &
                                               bse_iterdiag = 1

   ! BSE level of approximation to the ABBA matrix (i.e. with/without TDA)
   INTEGER, PARAMETER, PUBLIC               :: bse_tda = 0, &
                                               bse_abba = 1, &
                                               bse_both = 2

   ! BSE level of approximation to the screening
   INTEGER, PARAMETER, PUBLIC               :: bse_screening_w0 = 0, &
                                               bse_screening_tdhf = 1, &
                                               bse_screening_rpa = 2, &
                                               bse_screening_alpha = 3

   ! BSE iterative abortion condition
   INTEGER, PARAMETER, PUBLIC               :: bse_iter_en_cond = 0, &
                                               bse_iter_res_cond = 1, &
                                               bse_iter_both_cond = 2
   ! SIGMA_functional parameters selection
   INTEGER, PARAMETER, PUBLIC               :: sigma_none = 0, &
                                               sigma_PBE_S2 = 1, &
                                               sigma_PBE0_S1 = 2, &
                                               sigma_PBE0_S2 = 3, &
                                               sigma_PBE_S1 = 4

   ! SMEAGOL interface
   INTEGER, PARAMETER, PUBLIC               :: smeagol_bulklead_leftright = 1, &
                                               smeagol_bulklead_left = 2, &
                                               smeagol_bulklead_right = 3

   INTEGER, PARAMETER, PUBLIC               :: smeagol_gridmethod_traditional = 0, &
                                               smeagol_gridmethod_adaptive = 1

   INTEGER, PARAMETER, PUBLIC               :: smeagol_integraltype_gauss_legendre = 1, &
                                               smeagol_integraltype_gauss_chebyshev = 2

   INTEGER, PARAMETER, PUBLIC               :: smeagol_runtype_bulktransport = 0, &
                                               smeagol_runtype_emtransport = 1

END MODULE input_constants
