/* 'checkit_tiff' is a conformance checker for baseline TIFFs
 *
 * author: Andreas Romeyke, 2015-2022
 * licensed under conditions of libtiff
 * (see http://libtiff.maptools.org/misc.html)
 *
 */

#ifndef VALIDATE_ICC
#define VALIDATE_ICC
#include <string.h>
#include <assert.h>
#include <stdio.h>

#define ICC_IS_VALID 0
#define ICC_ERROR_OFFSET 100

typedef enum {
  icc_profile_1v43_2010,
  icc_profile_v240_v430,
  icc_profile_v500_2016,
} icc_profile_version_t;

typedef enum {
  icc_is_valid                       = ICC_IS_VALID,
  icc_error_profileclass             = ICC_ERROR_OFFSET, /* profile class ('%s'), should be one of following strings for device classes: 'scnr', 'mntr', 'prtr' or for profile classes: 'link', 'spac', 'abst', 'nmcl'" */
  icc_error_colorspacedata           = ICC_ERROR_OFFSET + 1, /* colorspace data ('%s'), should be one of following strings: 'XYZ ' 'Lab ' 'Luv ' 'YCbr' 'Yvx ' 'RGB ' 'GRAY' 'HSV ' 'HLS ' 'CMYK' 'CMY ' '2CLR' '3CLR' '4CLR' '5CLR' '6CLR' '7CLR' '8CLR' '9CLR' 'ACLR' 'BCLR' 'CCLR' 'DCLR' 'ECLR' 'FCLR'" */
  icc_error_connectionspacedata      = ICC_ERROR_OFFSET + 2, /* "connection space data ('%s') should be one of following strings: 'XYZ ' 'Lab '" */
  icc_error_primaryplatformsignature = ICC_ERROR_OFFSET + 3, /* primary plattform signature ('%s') should be empty or one of following strings: 'APPL', 'MSFT', 'SGI ', 'SUNW', 'TGNT' */
  icc_error_header_1v43_2010         = ICC_ERROR_OFFSET + 4, /* Invalid ICC profile 1v43_2010, see http://www.color.org/specification/ICC1v43_2010-12.pdf for details  */
  icc_error_header_v240_v430         = ICC_ERROR_OFFSET + 5, /* Invalid ICC profile ICC.1:2001-04, see http://www.color.org/ICC_Minor_Revision_for_Web.pdf for details */
  icc_error_header_generic           = ICC_ERROR_OFFSET + 6, /* size < 10 */
  icc_error_preferredcmmtype         = ICC_ERROR_OFFSET + 7, /* preferred cmmtype ('%s') should be empty or (possibly, because ICC validation is alpha code) one of following strings: 'ADBE' 'ACMS' 'appl' 'CCMS' 'UCCM' 'UCMS' 'EFI ' 'FF  ' 'EXAC' 'HCMM' 'argl' 'LgoS' 'HDM ' 'lcms' 'KCMS' 'MCML' 'WCS ' 'SIGN' 'RGMS' 'SICC' 'TCMM' '32BT' 'WTG ' 'zc00'" */
  icc_error_committed_size_differs   = ICC_ERROR_OFFSET + 8, /* committed ICC size differs from encoded profilesize */
  icc_error_header_v500_2016         = ICC_ERROR_OFFSET + 9, /* Invalid ICC profile ICC.1:2001-04, see  http://www.color.org/iccmax/ICC.2-2016-7.pdf for details */
  icc_error_header_version_undefined = ICC_ERROR_OFFSET +10, /* if header version was undefined yet */
  icc_error_header_version_outdated  = ICC_ERROR_OFFSET +11, /* if header version was outdated */
  icc_error_badprofile_id            = ICC_ERROR_OFFSET +12, /* if md5 missmatch */
  icc_error_not_ascii                = ICC_ERROR_OFFSET +13, /* if text fields uses 8bit */
  icc_error_profile_description_tag  = ICC_ERROR_OFFSET +14, /* if profile description tag has wrong length */
  icc_error_found_by_IccProfLib      = ICC_ERROR_OFFSET +90, /* generic error by IccProfLib */
  icc_hard_error_found_by_IccProfLib = ICC_ERROR_OFFSET +91, /* generic hard error by IccProfLib */
  icc_should_not_occur               = ICC_ERROR_OFFSET +99,
} icc_returncode_t;

icc_returncode_t parse_icc(unsigned long iccsize, const char* iccdata, unsigned long errsize, char * errorneous_value);
#ifdef HAVE_IccProfLib

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
// C++ Code
icc_returncode_t validate_icc_with_IccProfLib(unsigned long iccsize, const char* iccdata, unsigned long errsize, char * errorneous_value);
#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* HAVE_IccProfLib */
#endif