from twisted.trial import unittest
import os, sets, email

from MailPing import mail, maildir, fileutil

class TestMail(unittest.TestCase):
    def setUp(self):
        self.maildir = self.mktemp()
        os.mkdir(self.maildir)
        os.mkdir(os.path.join(self.maildir, 'cur'))
        os.mkdir(os.path.join(self.maildir, 'new'))
        os.mkdir(os.path.join(self.maildir, 'tmp'))

    def testGetID_Simple(self):
        fileutil.writeFile(os.path.join(self.maildir, 'new',
                                        '42.P34Q0.testhost'),
                           '''\
From: Foo Bar <foo.bar@thud.invalid>
To: Mr. You <you@destination.invalid>
Subject: Mail ping f0e241677fa9f2e0d7884776e4a6846b19703e6d
Reply-To: admin@foo.invalid

content is just junk
''')

        seen = sets.Set()
        def _store(path, filename):
            x = mail.getID(os.path.join(path, filename))
            seen.add((filename, x))
        maildir.process(os.path.join(self.maildir),
                        _store)
        self.assertEquals(seen,
                          sets.Set([
            ('42.P34Q0.testhost', 'f0e241677fa9f2e0d7884776e4a6846b19703e6d'),
            ]))


    def testGetID_NoSubject(self):
        fileutil.writeFile(os.path.join(self.maildir, 'new',
                                        '42.P34Q0.testhost'),
                           '''\
From: Foo Bar <foo.bar@thud.invalid>
To: Mr. You <you@destination.invalid>
Reply-To: admin@foo.invalid

content is just junk
''')

        seen = sets.Set()
        def _store(path, filename):
            x = mail.getID(os.path.join(path, filename))
            seen.add((filename, x))
        maildir.process(os.path.join(self.maildir),
                        _store)
        self.assertEquals(seen,
                          sets.Set([
            ('42.P34Q0.testhost', None),
            ]))

    def testGetID_InvalidSubject(self):
        msgpath = os.path.join(self.maildir, 'new',
                               '42.P34Q0.testhost')
        fileutil.writeFile(msgpath,
                           '''\
From: Foo Bar <foo.bar@thud.invalid>
To: Mr. You <you@destination.invalid>
Subject: SPAM SPAM SPAM!
Reply-To: admin@foo.invalid

content is just junk
''')

        seen = sets.Set()
        def _store(path, filename):
            x = mail.getID(os.path.join(path, filename))
            seen.add((filename, x))
        maildir.process(os.path.join(self.maildir),
                        _store)
        self.assertEquals(seen,
                          sets.Set([
            ('42.P34Q0.testhost', None),
            ]))

    def testSetID_OK(self):
        msg = email.message_from_string('''\
From: Foo Bar <foo.bar@thud.invalid>
To: Mr. You <you@destination.invalid>
Reply-To: admin@foo.invalid

content is just junk
''')
        mail.setID(msg, 'f0e241677fa9f2e0d7884776e4a6846b19703e6d')
        self.assertEquals(msg.as_string(),
                          '''\
From: Foo Bar <foo.bar@thud.invalid>
To: Mr. You <you@destination.invalid>
Reply-To: admin@foo.invalid
Subject: Mail ping f0e241677fa9f2e0d7884776e4a6846b19703e6d

content is just junk
''')

    def testSetID_HasSubject(self):
        msg = email.message_from_string('''\
From: Foo Bar <foo.bar@thud.invalid>
Subject: I should not be here
To: Mr. You <you@destination.invalid>
Reply-To: admin@foo.invalid

content is just junk
''')
        self.assertRaises(mail.TemplateMustNotHaveFieldError,
                          mail.setID,
                          msg, 'f0e241677fa9f2e0d7884776e4a6846b19703e6d')
        try:
            mail.setID(msg, 'f0e241677fa9f2e0d7884776e4a6846b19703e6d')
        except mail.TemplateMustNotHaveFieldError, e:
            self.assertEquals(
                str(e),
                "Probe email template must not have this field: "
                + "'Subject', 'I should not be here'")
